' Akalabeth by Richard Garriott, 1979, for the Apple ][ computer
' Ported to the CMM2 by Vegipete, Dec 2021
' Title Image from the DOS port
'
' Akalabeth is seen as the first game in the Ultima Series.
'
' 98 GOTOs in original Applesoft version
'   of which 9 are computed (ON {expr} GOTO linenum1,linenum2,...)
' 38 implied GOTOs (IF THEN [GOTO] linenum)
' 15 GOSUBs

' pw(0) = number of FOOD
' pw(1) = number of RAPIER
' pw(2) = number of AXE
' pw(3) = number of SHIELD
' pw(4) = number of BOW AND ARROWS
' pw(5) = number of MAGIC AMULET
dim pw(5) ' quantity of items held

' c(0) , cn$(0) = "Hit points....."
' c(1) , cn$(1) = "Strength......."
' c(2) , cn$(2) = "Dexterity......"
' c(3) , cn$(3) = "Stamina........"
' c(4) , cn$(4) = "Wisdom........."
' c(5) , cn$(5) = "Gold..........."
dim cn$(5) length 16 ' names of player statistics
dim c(5)  ' level of statistics

dim m$(10) length 16  ' names of evil enemies

dim ml%(10,1)   ' x,y coords of monsters 1 to 10 in dungeon
dim mz%(10,1)
dim w$(5)       ' list of names of items (food, weapons)
dim dng%(10,10) ' map of dungeon, created each time level entered
dim te%(20,20)  ' map of ground terain
dim xx%(10),yy%(10),per%(10,3),ld%(10,5)
dim cd%(10,3),ft%(10,5),ladder%(10,3)
dim integer l1,l2,r1,r2,t1,t2,b1,b2   ' dungeon drawing variables

dim seed%

pn$ = ""    ' no player name at start

dim debugview = 0 ' draw plan view of dungeon if not zero - for debugging

DefineFont #8   ' 24 x 8 glyph of arrow keys
  01210818 70040020 00A90481 C4E72384 20950021 00200E81 00000004
End DefineFont

mode 3 : pause 2500 ' let monitor sync
cls

' Page usage:
' 0 - what we see
' 1 - constructions
' 3 - curtain animation
' 6 - title screen

' display title screen for player's enjoyment
page write 6
cls
chdir mm.info$(path)
load png "Akalabeth_Title.png"
page write 0
for x = 0 to MM.HRES/2  ' display title screen
  blit MM.HRES/2-x,0,MM.HRES/2-x,0,1,MM.VRES,6
  blit MM.HRES/2+x,0,MM.HRES/2+x,0,1,MM.VRES,6
  pause 15
next x

timer = 0
do  ' show title for a while, or until player presses any key
  if timer > 5000 then exit do
  if inkey$ <> "" then exit do
loop

' shrink image out of sight
page write 1
for x = 2 to 158 step 2
  y = x * 200/320
  cls
  image resize_fast 0, 0, MM.HRES, MM.VRES, x, y, MM.HRES-2*x, MM.VRES-2*y,6
  page copy 1,0,B
next x
page write 0

CreateCharacter  ' and visit The Adventure Shop, also seeds pseudo random numbers

cls
text MM.HRES/2,   0, "Welcome to Akalabeth",CT,1, , rgb(yellow)
text MM.HRES/2,  12, "World of Doom!",CT,1, , rgb(yellow)

DrawWorldTiles  ' pre-generate overhead view tiles

' random terrain map
math set 0,te%()  ' erase the world to empty plains
for x = 0 to 20   ' make border mountains
  te%(x,0)=1 : te%(0,x)=1 : te%(x,20)=1 : te%(20,x)=1
next x

' random inside of 19 x 19 terrain tiles
' 0=plains, 1=mountains, 2=tree, 3=town, 4=dungeon, 5=LBcastle
for x = 1 to 19
  for y = 1 to 19
    te%(x,y) = int(rand(1) ^ 5 * 4.5)
    if te%(x,y) = 3 and rand(1) > .5 then te%(x,y)=0
  next y
next x

' random place lord british castle
te%(int(rand(1) *19 + 1),int(rand(1) * 19 + 1)) = 5

' random place for town/shop, which is also player starting location
tx = int(rand(1) * 19 + 1) : ty = int(rand(1) * 19 + 1) : te%(tx,ty) = 3

' calculate a bunch of screen coordinates for drawing perspective
' Notes:
' The Apple ][ screen was 280 pixels wide by 192 pixels high. Akalabeth used
' the mixed graphics mode, consisting of 280x160 pixel HiRes display above
' four lines of text. So various 'magic numbers' refered to specific screen
' locations: 279 = right edge of screen, 159 = bottom edge of graphics area
' 139 = right edge of left half, 140 = left edge of right half
' 79 = bottom edge of top half, 80 = top edge of bottom half
'
' I've used CMM2 Mode 3, which is 320 pixels wide x 200 pixels high.
' Rows 0 to 83 form the top half, 84 to 167 the bottom half of the graphics region
' and rows 168 to 199 will be used for the 4 lines of text.
xx%(0) = MM.HRES/2 - 1    ' original 139, replaced with 159
yy%(0) = MM.VRES/2 - 17   ' original  79, replaced with  83

for x = 2 to 20 step 2
  xx%(x/2) = int(atn(1/x) / atn(1) * MM.HRES/2 +.5)
  yy%(x/2) = int(xx%(x/2) * 4 / 7)
  per%(x/2,0) = xx%(0) - xx%(x/2) : per%(x/2,1) = xx%(0) + xx%(x/2)
  per%(x/2,2) = yy%(0) - yy%(x/2) : per%(x/2,3) = yy%(0) + yy%(x/2)
next x

per%(0,0) = 0 : per%(0,1) = MM.HRES-1 : per%(0,2) = 0 : per%(0,3) = MM.VRES-33

for x = 1 to 10
  cd%(x,0) = xx%(0)-xx%(x)/ 3
  cd%(x,1) = xx%(0)+xx%(x)/ 3
  cd%(x,2) = yy%(0)-yy%(x)*.7
  cd%(x,3) = yy%(0)+yy%(x)
next x

for x=0 to 9
  ld%(x,0) = (per%(x,0)*2+per%(x+1,0))/3
  ld%(x,1) = (per%(x,0)+2*per%(x+1,0))/3
  wrk = ld%(x,0)-per%(x,0)
  ld%(x,2) = per%(x,2)+wrk*4/7
  ld%(x,3) = per%(x,2)+2*wrk*4/7
  ld%(x,4) = (per%(x,3)*2+per%(x+1,3))/3
  ld%(x,5) = (per%(x,3)+2*per%(x+1,3))/3
  ld%(x,2) = ld%(x,4)-(ld%(x,4)-ld%(x,2))*.8
  ld%(x,3) = ld%(x,5)-(ld%(x,5)-ld%(x,3))*.8
  if ld%(x,3) = ld%(x,4) then ld%(x,3) = ld%(x,3)-1
next x

for x=0 to 9
  ft%(x,0) = xx%(0)-xx%(x)/3
  ft%(x,1) = xx%(0)+xx%(x)/3
  ft%(x,2) = xx%(0)-xx%(x+1)/3
  ft%(x,3) = xx%(0)+xx%(x+1)/3
  ft%(x,4) = yy%(0)+(yy%(x)*2+yy%(x+1))/3
  ft%(x,5) = yy%(0)+(yy%(x)+2*yy%(x+1))/3
next x

for x=0 to 9
  ladder%(x,0) = (ft%(x,0)*2+ft%(x,1))/3
  ladder%(x,1) = (ft%(x,0)+2*ft%(x,1))/3
  ladder%(x,3) =  ft%(x,4)
  ladder%(x,2) = per%(0,3) - ladder%(x,3)
next x

' CMM2 is so fast that we can actually slow things down to read intro.
for x=0 to 20
  text MM.HRES/4 + x * MM.HRES/40,  24, ".",CT,1, , RGB(GREEN)
  pause 50
next x

'===================
' test stuff here

' end of test
'===================

dngLevel = 0   ' 0 = on the surface, 1-10 = dungeon level
redraw = 1

do  ' game loop
  if redraw then DrawPlayerView   ' only when something changes
  
  DisplayPlayerStats  ' display player stats bottom right, every time

  ScrollCommands  0, "Command (    SATXP)?"
  text 53,200, "!", LB, 8   ' display arrow symbols in custom font

  ' Commands are:
  ' ARROW keys to move N/S/E/W or Forward/Turn Around/Turn Left/Turn Right
  ' S=Stats A=Attack T=Pass P=Toggle Pause Mode
  ' X=climb (up or down) ladder/enter town/enter castle
  kp = asc(WaitUserKey(""))  ' wait for any player keypress

  select case kp
    case 128      ' North/Forward
      if dngLevel then
        MoveForward
      else
        MoveNSEW("North",0,-1)
      endif
    case 130      ' West/Turn Left
      if dngLevel then
        TurnLeft
      else
        MoveNSEW("West",-1,0)
      endif
    case 131      ' East/Turn Right
      if dngLevel then
        TurnRight
      else
        MoveNSEW("East",+1,0)
      endif
    case 129      ' South/Turn Around
      if dngLevel then
        TurnAround
      else
        MoveNSEW("South",0,+1)
      endif
    case asc("S")   ' Statistics
      page copy 0,2     ' save image
      DisplayPlayerInfo ' show something else
      
      ' display current task - added by Vegipete
      qq$ = "Quest: "
      if task < 0 then
        qq$ = qq$ + "return to Lord British."
      elseif task > 0 then
        qq$ = qq$ + "kill a"
        if task = 4 then qq$ = qq$ + "n"
        qq$ = qq$ + " " + m$(task) + "."
      else
        qq$ = qq$ + "visit Lord British."
      endif
      text MM.HRES/2,80,qq$,CM
      
      print @(70,126);   ' position cursor
      WaitForKeypress
      page copy 2,0     ' restore image
    case asc("A")   ' Attack!!!!
      if dngLevel then
        PlayerAttacks
      else
        ' no attacks on surface
      endif
    case asc("T"),asc(" ")   ' Pass - no action
      if dngLevel then
      else
      endif
    case asc("X")   ' Enter/Climb
      if dngLevel then
        TryClimbLadder  ' up or down (or not at all), depending on where player is
      else
        EnterSomething  ' try enter town, dungeon or castle
      endif
    case asc("P")   ' Pause
      ' doesn't serve much purpose, ignored
    case asc("Q")   ' Quit
      ScrollCommands  0, "Quit! Are you sure?"
      if WaitUserKey("") = "Y" then end
    case else       ' unknown command
      if kp = asc("#") then
        debugview = debugview = 0 ' hidden command to toggle plan view in dungeon
        if dngLevel then redraw = 1
      endif
      RewriteCommands 47, "  Huh?      "
  end select

  pw(0) = max(0, pw(0) - 1 + sgn(dngLevel) * .9)  ' use 1 food per turn on surface
                                                  ' 0.1 food per turn in dungeon
  if pw(0) = 0 then
    c(0) = 0    ' Hitpoints = 0
    ScrollCommands 10, "You have starved!!!!!"
  else
    pw(0) = int(pw(0)*10) / 10  ' round it off neatly
  endif
  
  if c(0) = 0 then DeathCharacter  ' Hitpoints gone
  
  if dngLevel > 0 then
    MonsterHandle   ' monsters do their thing - attack, steal, move, whatever
    if c(0) = 0 then DeathCharacter
  endif

loop

end

'====================================================================
' Scroll the text command window up one line and display text at very bottom
' window is located at (0,168) to (259,199)
' Handles long strings gracefully - well, not yet...
sub ScrollCommands xpos,t$
  blit 0,176,0,168,259,24   ' copy bottom 3 lines of text up one line
  box  0,192,259,8,0,0,0    ' erase bottom line of text
  RewriteCommands xpos,t$
end sub

'==================
' Add/change text on bottom line without scrolling
sub RewriteCommands xpos,t$
  text xpos,200, t$, LB
end sub

'  '==================
'  ' this time, actually print the text
'  UB_1$ = msg$
'  y = 0
'  do
'    do while left$(UB_1$,1) = " "
'      UB_1$ = mid$(UB_1$,2)   ' hack off leading spaces
'    loop
'    if UB_1$ = "" then exit do
'    SplitString(UB_1$, UB_2$, UB_cwidth)
'    text UB_x+8,UB_y+y*MM.INFO(FONTHEIGHT)+10,UB_1$,"LT",,,UB_cols(2),UB_cols(1)
'    y = y + 1
'    UB_1$ = UB_2$
'    if y > UB_lines-1 then exit do    ' don't overflow message box - shouldn't happen
'  loop
'
''==================
'' SplitString : used for word wrapping
'' Split 'in$' at a deliminator in 'd$' such that 'in$' is a long as possible but not
'' longer than 'size'. Return the left over string, without the deliminator, in 'tail$'
'' CHR$(13) allows for line breaks
'sub SplitString(in$, tail$, size, d$)
'  local integer sp
'  local string head$,delim$
'
'  tail$ = ""
'  sp = instr(in$, chr$(13))
'  if sp > 0 and sp < size then ' is there a line break in the first size characters?
'    tail$ = mid$(in$,sp+1)
'    in$ = left$(in$,sp-1)
'    exit sub
'  endif
'  if size >= len(in$) then exit sub
'  delim$ = d$
'  if delim$ = "" then delim$ = " "
'  sp = size + 1
'  do
'    'if instr(delim$, mid$(in$,sp,1)) then
'    if delim$ = mid$(in$,sp,1) then
'      tail$ = mid$(in$,sp+1)  'right$(in$,len(in$) - sp)
'      head$ = left$(in$,sp)
'      exit do
'    endif
'    sp = sp - 1
'  loop until sp = 0
'  if sp then
'    in$ = head$
'  else    ' no deliminator found
'    tail$ = mid$(in$,size+1)    ' break at message box width
'    in$ = left$(in$,size)
'  endif
'  do while instr(delim$, right$(in$,1)) ' hack off trailing spaces
'    in$ = mid$(in$, 1, len(in$) - 1)
'  loop
'end sub

'====================================================================
' Wait for the user to enter a key. Converted to UPPER case
' Only keys in the pattern are accepted. Any key is fine if the pattern is empty.
function WaitUserKey(pat$) as string
  local qq$
  do while inkey$ <> "" : loop              ' clear any pre-existing keypresses
  do                                        ' wait for user to enter something
    qq$ = ucase$(inkey$)
    if qq$ <> "" and pat$ = "" then exit do ' no pattern so accept anything
  loop until instr(ucase$(pat$),qq$)        ' keep looping until it's a valid key
  WaitUserKey = qq$
end function

' Print message at cursor location and wait for key press
sub WaitForKeypress
  do while inkey$ <> "" : loop    ' clear any keystrokes
  print "   Press any key to continue...";
  do : loop until inkey$ <> ""    ' wait for keystroke
end sub

'====================================================================
' Move to a new location
sub MoveNSEW(t$,dx,dy)
  RewriteCommands 47,left$("  "+t$+"       ",13)
  if te%(tx+dx,ty+dy) = 1 then
    ScrollCommands 10, "You can't pass the mountains"
  else
    tx = tx + dx
    ty = ty + dy
    redraw = 1
  endif
end sub

sub MoveForward     ' 11500 
    RewriteCommands 47, "  Forward    "

  if dng%(px+pdx,py+pdy)<>1 and dng%(px+pdx,py+pdy)<10 then   ' move if nothing in the way
    px = px+pdx
    py = py+pdy
    redraw = 1

    if dng%(px,py)=2 then
      ScrollCommands 10, "AAARRRGGGHHH!!! A Trap!"
      c(0)= max(0, c(0) - int(rand(1) * dngLevel + 3))    ' loose some hit points
      dngLevel=dngLevel+1   ' fall down
      CreateDungeon
      mr=1  ' purpose unknown (appears 3 times, never used)
      ScrollCommands 10, "Falling to level "+str$(dngLevel)
    elseif dng%(px,py)=5 then   ' a chest
      dng%(px,py)=0     ' erase the chest from dungeon
      z = int(rand(1) * 5 * dngLevel + dngLevel)    ' random amount of gold in chest
      if z = 1 then
        ScrollCommands 10, "GOLD!!! One piece."
      else
        ScrollCommands 10, "GOLD!!! "+str$(z)+" pieces!"
      endif
      c(5) = c(5) + z   ' yah, more gold
      z = int(rand(1) * 6)    ' and random loot in chest
      ScrollCommands 10, "  And a " + w$(z)
      pw(z) =  pw(z) + 1
    endif
  else
    RewriteCommands 107, "- Ooof"   ' bumped into something
  endif
end sub

sub TurnRight
  RewriteCommands 47, "  Turn right "
  if pdx <> 0 then
    pdy=pdx : pdx=0
  else
    pdx=-pdy : pdy=0
  endif
  redraw = 1
end sub

sub TurnLeft
  RewriteCommands 47, "  Turn left  "
  if pdx <> 0 then
    pdy=-pdx : pdx=0
  else
    pdx=pdy : pdy=0
  endif
  redraw = 1
end sub

sub TurnAround
  RewriteCommands 47, "  Turn around"
  pdx=-pdx : pdy=-pdy
  redraw = 1
end sub

'====================================================================
sub EnterSomething  ' 15000
  if te%(tx,ty)=3 then ' city
    DisplayPlayerInfo
    GoShopping
    redraw = 1
  elseif te%(tx,ty)=4 and dngLevel = 0 then ' dungeon   ' 15100 
    box 0,0,MM.HRES,MM.VRES-32,1,0,0  ' erase graphics region
    RewriteCommands 47, "  Enter dungeon"
    dngLevel = 1
    CreateDungeon
    pdx=1 : pdy=0   ' initial direction
    px=1 : py=1   ' initial position
    lk=0    ' hit points gained while in dungeon
  elseif te%(tx,ty)=5 then ' castle     ' 15150
    VisitLordBritish
  else  ' nothing here to enter
    RewriteCommands 47, "  Huh?      "
  endif
end sub

'====================================================================
sub TryClimbLadder
  if dng%(px,py) = 7 or dng%(px,py) = 9 then ' ladder down, trapdoor in floor
    dngLevel=dngLevel+1
    CreateDungeon
    mr=1
    RewriteCommands 47, "  Down to level " + str$(dngLevel)
  elseif dng%(px,py) = 8 then ' ladder up
    dngLevel = dngLevel - 1
    if dngLevel = 0 then
      RewriteCommands 47, "  Leave dungeon."
      ScrollCommands  8, "Thou hast gained " + str$(lk) + " hit points."
      c(0)=c(0)+lk
      redraw = 1
    else
      CreateDungeon
      mr=1
      RewriteCommands 47, "  Up to level " + str$(dngLevel)
    endif
  else    ' not at ladder
    RewriteCommands 47, "  Huh?      "
  endif
end sub

'====================================================================
' attack something - such unspeakable violence!
sub PlayerAttacks
  mn = 0    ' monster being attacked
  dam = -1  ' damage done by weapon
  far = 0   ' set to 1 for distant attack - bow&arrow or thrown axe
  RewriteCommands 47, "  Attack!   "
  do ' keep looping until valid weapon, hands or cancel selected
    ScrollCommands 1, "Which weapon? (R/A/S/B/M) or (H)ands"
    qq$ = WaitUserKey("abhmrs ")   ' space bar to cancel
    select case qq$
      case "A"  ' axe
        if pw(2) < 1 then
          ScrollCommands 10, "Axe - not owned."
          continue do
        endif
        ScrollCommands 10, "Axe - to swing or throw? (S/T)"
        qq$ = WaitUserKey("st")
        if qq$ = "T" then
          'RewriteCommands 28," - to throw                "
          RewriteCommands 64,"throw                "
          pw(2) = pw(2) - 1    ' use up an axe
          far = 1
        else
          'RewriteCommands 28," - to swing                "
          RewriteCommands 64,"swing                "
        endif
        dam = 5
        exit do
      case "B"  ' bow
        if pw(4) < 1 then
          ScrollCommands 10, "Bow - not owned."
          continue do
        else
          if pt$ = "M" then
            ScrollCommands 10, "Mages can't use bows!"
            continue do
          endif
        endif
        ScrollCommands 10, "Bow"
        dam = 4
        far = 1
        exit do
      case "H"  ' hands
        ScrollCommands 10, "Hands"
        dam = 0
        exit do
      case "M"  ' magic amulet
        if pw(5) < 1 then
          ScrollCommands 10, "Magic amulet - not owned."
          continue do
        endif
        ScrollCommands 10, "Magic amulet"
        'do magic stuff here
        if pt$ = "F" then   ' fighter can't control magic
          q = int ( rand (1) * 4 + 1)
        else
          ScrollCommands 1, "1=ladder-up  3=kill"
          ScrollCommands 1, "2=ladder-dn  4=bad??   Choose action"
          q = val(WaitUserKey("1234"))
          if rand(1)>.75 then
            ScrollCommands 10, "Last charge on this amulet!"
            pw(5) = pw(5)-1
          endif
        endif
        select case q
          case 1
            ScrollCommands 10, "ladder up"
            dng%(px,py) = 8
            redraw = 1
            exit sub
          case 2
            ScrollCommands 10, "ladder down"
            dng%(px,py) = 7
            redraw = 1
            exit sub
          case 3
            ScrollCommands 10, "magic attack"
            dam = 10 + dngLevel
            far = 1     ' distant attack
          case 4
            q1 = int(rand(1) * 3 + 1)   ' random bad?? action happens...
            select case q1
              case 1
                ScrollCommands 1, "You have been turned into a toad!"
                for z2 = 1 to 4:c(z2) = 3:next z2
                exit sub
              case 2
                ScrollCommands 1, "You've been turned into a lizard man"
                for z2 = 0 to 4:c(z2) = int(c(z2)*2.5):next z2
                exit sub
              case 3
                ScrollCommands 1, "Backfire!"
                c(0) = c(0) / 2   ' loose half of hit points
                exit sub
            end select
        end select
        exit do
      case "R"  ' rapier
        if pw(1) < 1 then
          ScrollCommands 10, "Rapier - not owned."
          continue do
        else
          if pt$ = "M" then
            ScrollCommands 10, "Mages can't use rapiers!"
            continue do
          endif
        endif
        ScrollCommands 10, "Rapier"
        dam = 5
        exit do
      case "S"  ' shield
        if pw(3) < 1 then
          ScrollCommands 10, "Shield - not owned."
          continue do
        endif
        ScrollCommands 10, "Shield"
        dam = 1
        exit do
      case " "  ' cancel
        exit sub
    end select
  loop  ' until weapon chosen

  if far then     ' distant target - bow or thrown axe or magic
    for y=1 to 5  ' weapon travels up to 5 cells away
      if px+pdx*y < 1 or px+pdx*y > 9 or py+pdy*y > 9 or py+pdy*y < 0 then  ' hit outside wall
        ScrollCommands 10, "You missed."
        exit for
      else
        mn = dng%(px + pdx * y,py + pdy * y)  ' is there a monster there?
        mn = int(mn/10)
        if mn > 0 then ' possible hit
          TryHitMonster
          exit for    ' subroutine is done
        endif
      endif
    next y    ' weapon keeps going
  else    ' near combat
    mn = dng%(px + pdx * 1,py + pdy * 1)  ' is there a monster there?
    mn = int(mn/10)
    if mn > 0 then ' possible hit
      TryHitMonster
    else
      ScrollCommands 10, "Nothing there."
    endif
  endif
end sub

' Perform attack on monster.
' Could be a miss, could be a hit.
' If it is a hit, the monster looses hit points.
' If the monster's hit points are all gone, it perishes and the player is rewarded.
' If the vanquished monster is the target of the player's quest, all the better.
sub TryHitMonster
  if c(2)-rand(1)*25 < mn+dngLevel then
    ScrollCommands 10, "You missed."
  else
    dam = rand(1) * dam + c(1) / 5
    mz%(mn,1) = mz%(mn,1) - dam
    ScrollCommands 2, "Hit!!! "+m$(mn)+"'s hit points="+str$(max(mz%(mn,1),0))  ' no negative numbers
    if mz%(mn,1)<1 then   ' monster be dead.
      ScrollCommands 1, "Thou hast killed a " + m$(mn)
      dam = int(mn+dngLevel) ' reuse variable for amount of gold bounty
      ScrollCommands 1, "Thou shalt receive " + str$(dam) + " gold."
      c(5) = int(c(5)+dam)
      dng%(ml%(mn,0),ml%(mn,1)) = dng%(ml%(mn,0),ml%(mn,1))-10*mn ' remove monster from dungeon
      mz%(mn,0) = 0
      lk=lk+int(mn*dngLevel/2)    ' total hit points gained while in dungeon
      if mn=task then task=-task    ' Lord British will be pleased
      DrawDungeonView     ' update view - monster is gone
      DisplayPlayerStats  ' display player stats bottom right
      'ScrollCommands 21, "Press any key to continue..."
      'qq$ = WaitUserKey("")
    endif
  endif
end sub

'====================================================================
' Visit Lord British
' On the very first visit, Lord British asks the player's name and sends
' him on his first quest. On each subsequent return after a successful
' quest, the next higher monster is chosen for the next quest. After the
' top monster has been slain, Knighthood is aquired and play at the next
' higher level is suggested. Success at the highest difficulty level
' results in a message to call the game's publisher.
sub VisitLordBritish
  cls
  if pn$ = "" then    ' player name
    print
    print
    print "Welcome peasant into the halls of the mighty"
    print "Lord British. Herein thou may choose to dare"
    print "battle with the evil creatures of the depths,"
    print "for great reward!"
    print
    print "What is thy name peasant";:input pn$
    print "Doth thou wish for grand adventure y/n";:input qq$
  
    if ucase$(left$(qq$,1)) <> "Y" then
      print
      print "Then leave and be gone!"
      pn$ = ""
      print
      WaitForKeypress
    else
      print
      print "Good! Thou shalt try to become a knight!!!"
      print
      print "Thy first task is to go into the dungeons and to"
      print "return only after killing a ";
      'task=int(c(4)/3) ' original, no good if wisdom is higher than 30
      task = c(4) mod 3 + 1
      print m$(task);
      print "."
      print
      print "Go now upon this quest, and may lady luck be fair "
      print "unto you........also I, British, have increased"
      print "each of thy attributes by one!"
      print
      for x = 0 to 5 : inc c(x) : next
      WaitForKeypress
    endif
    
  else    ' returning to the castle
    if task > 0 then    ' task not complete
      print
      print
      print pn$;", why hast thou returned?"
      print "Thou must kill a";
      if task = 4 then print "n";
      print " ";
      print m$(task);
      print "."
      print "Go now and complete thy quest!"
      print
      WaitForKeypress
    else
      print : print : print
      print "Aahh!!.....";pn$
      print
      print "Thou hast acomplished thy quest!"
      if task = -10 then    ' defeated senior monster
        print : print : print
        pn$="Lord "+pn$
        print " ";pn$;","
        print "    thou hast proved thyself worthy of knighthood!"
        print "Continue play if thou doth wish, but thou hast "
        print "acomplished the main objective of this game..."
        if skill = 10 then
          print
          print "   ...call California Pacific Computer"
          print "at (415)-569-9126 to report this amazing feat!"
        else
          print
          print "   Now maybe thou art foolhearty enough to try"
          print "difficulty level ";skill + 1
          inc skill
          for x = 0 to 5 : inc c(x) : next
        endif
        print
        WaitForKeypress
      else    ' defeated lesser monster so advance to next one
        print "Alas, this is not enough to become a knight."
        task =  abs(task) + 1
        print
        print "Now thou must kill a";
        if task = 4 then print "n";
        print " ";
        print m$(task);
        print "."
        print
        print "Go now upon this quest, and may lady luck be fair "
        print "unto you........also I, British, have increased"
        print "each of thy attributes by one!"
        print
        for x = 0 to 5 : inc c(x) : next
        WaitForKeypress
      endif
    endif
  endif
  cls ' box   0,MM.VRES-32,MM.HRES,32,1,0,0  ' erase text region
  redraw = 1
end sub

'====================================================================
' Draw what the player sees - either surface map or dungeon FPV
sub DrawPlayerView

  if dngLevel = 0 then
    DrawWorldView
  elseif dngLevel > 0 then
    DrawDungeonView
  endif

  redraw = 0
end sub

'====================================================================
'Draw the birds eye view of the world, centered on the player's location
sub DrawWorldView
  box 0,0,MM.HRES,MM.VRES-32,0,0,0  ' erase graphics region
  for y=-1 to 1
    for x=-2 to 2   ' Vegipete: added more viewable width
      if tx+x < 0 or tx+x > 20 then
        blit write 2,(x+1)*50+85,(y+1)*50+4   ' mountains outside of terrain array
      else
        blit write te%(tx+x,ty+y)+1,(x+1)*50+85,(y+1)*50+4
      endif
    next x
  next y
  line 158,79,162,79,1,RGB(WHITE):line 160,77,160,81,1,RGB(WHITE) ' mark player posn
end sub

' Pre-draw 6 landscape tiles and save for blitting
sub DrawWorldTiles
  page write 1
  
  cls
  DrawGrassItem
  blit read 1,0,0,50,50,1
  DrawTreeItem
  blit read 3,0,0,50,50,1
  
  cls
  blit write 1,0,0          ' redraw grass
  DrawDungeonEntrance
  blit read 5,0,0,50,50,1
  
  cls
  DrawMountainItem
  blit read 2,0,0,50,50,1
  
  cls
  DrawCityItem
  blit read 4,0,0,50,50,1
  
  cls
  DrawCastleItem
  blit read 6,0,0,50,50,1
  
  page write 0
end sub

' Draw grassy plains  - Vegipete embelishment
sub DrawGrassItem
  for grass = 1 to 50
    pixel 50*rnd,50*rnd,RGB(GREEN)
  next grass
end sub

' Draw mountain-lines
sub DrawMountainItem
  line 10,50,10,40,1,RGB(GREEN) ' foothills
  line  0,10,10,10,1,RGB(GREEN)
  line 50,10,40,10,1,RGB(GREEN)
  line  0,40,10,40,1,RGB(GREEN)
  line 10, 0,10,10,1,RGB(GREEN)
  line 40,10,40, 0,1,RGB(GREEN)
  line 40,40,40,50,1,RGB(GREEN)
  line 40,40,50,40,1,RGB(GREEN)

  line 10,10,10,20,1,RGB(BLUE)  ' upper reaches
  line 10,40,20,30,1,RGB(BLUE)
  line 40,30,40,40,1,RGB(BLUE)
  line 10,20,20,20,1,RGB(BLUE)
  line 30,20,30,10,1,RGB(BLUE)
  line 30,10,40,10,1,RGB(BLUE)
  line 20,30,40,30,1,RGB(BLUE)

  line 20,20,20,30,1,RGB(CYAN)  ' snow cap
  line 20,30,30,30,1,RGB(CYAN)
  line 30,30,30,20,1,RGB(CYAN)
end sub

' Draw a small box - represents a tree!
sub DrawTreeItem
  'box x1+20,y1+20,11,11,1,RGB(GRAY)
  circle 25,25,6,1,1,RGB(GREEN)   ' circle looks better, says I
end sub

' Draw city-lines
sub DrawCityItem
  box 10,10,11,11,1,RGB(BROWN)
  box 30,10,11,11,1,RGB(BROWN)
  box 10,30,11,11,1,RGB(BROWN)
  box 30,30,11,11,1,RGB(BROWN)
  box 20,20,11,11,1,RGB(ORANGE)
end sub

' Draw dungeon entry
sub DrawDungeonEntrance
  text 24,20,"U",CMI,5,1,RGB(RED)     ' upside down U to form arched entrance
  line 20,20,30,30,1,RGB(RED)
  line 20,30,30,20,1,RGB(RED)
end sub

' Draw castle-lines
sub DrawCastleItem
  box   5, 5,41,41,1,RGB(GRAY),RGB(GRAY)
  box  10,10,31,31,1,RGB(YELLOW),0
  line 10,10,40,40,1,RGB(YELLOW)
  line 10,40,40,10,1,RGB(YELLOW)
end sub

'====================================================================
' Display the player's stats bottom right
' Limit any displayed value to 999 or less but no less than 0
sub DisplayPlayerStats
  box 260,MM.VRES-32,60,32,1,0,0 'RGB(GREY),0  ' erase stats region

  text 291,168, "Food=", RT,,,RGB(GREEN)
  text 291,168, str$(min(pw(0),999)), LT,,,RGB(GREEN)
  text 291,176, "H.P.=", RT,,,RGB(GREEN)
  text 291,176, str$(min(c(0),999)), LT,,,RGB(GREEN)
  text 291,184, "Gold=", RT,,,RGB(GREEN)
  text 291,184, str$(min(c(5),999)), LT,,,RGB(GREEN)
  if dngLevel then
    text 297,192, "Depth=", RT,,,RGB(GREEN)
    text 297,192, str$(dngLevel), LT,,,RGB(GREEN)
  else
    text 303,192, "Outside", RT,,,RGB(GREEN)
  endif
end sub

'====================================================================
' Display the player's information
sub DisplayPlayerInfo

  cls
  text 50, 0, "Stats                   Weapons", LT
  for x = 0 to 5
    text  10,10+x*8,cn$(x), LT,,,,-1
    text 102,10+x*8,str$(c(x)) + "  ", LT,,,,-1
    text 195,10+x*8,str$(pw(x)), RT
    text 202,10+x*8,"- "+w$(x)  , LT,,,,-1
  next x
  
end sub

sub DisplayPrices
  text  20, 137, "Price", LT
  text 110, 137, "Damage", LT
  text 200, 137, "Item", LT
  
  for x = 0 to 5
    text 200,149+x*8,w$(x), LT,,,,-1
  next x
  
  text  20,149,"1 for 10" : text 110,149,"N/A"
  text  20,157,"8" : text 110,157,"1-10"
  text  20,165,"5" : text 110,165,"1-5"
  text  20,173,"6" : text 110,173,"1"
  text  20,181,"3" : text 110,181,"1-4"
  text  20,189,"15" : text 110,189,"?????"
end sub

'====================================================================
' Display a message upon death
sub DeathCharacter
  DisplayPlayerStats    ' ensure displayed values are up-to-date
  
  ' The Curtains Close - special fx by Vegipete
  page write 3 : cls
  for i = 100 to 360
    page write 3
    circle -100,0,i,2,1,RGB(BROWN)
    page write 0
    blit 0,0,0,0,MM.HRES/2,168,3,4
    blit 0,0,MM.HRES/2,0,MM.HRES/2,168,3,5
    pause 10
  next i

  colour RGB(YELLOW),RGB(BROWN)
  if len(pn$) > 22 then pn$=""
  if pn$ = "" then pn$ = "the peasant"
  pn$ = pn$ + " and his computer"
  print
  print
  print "               We mourn the passing of"
  print space$(26 - int(len(pn$) / 2)); pn$
  print
  print "        To invoke a miracle of ressurection,"
  print "                   press <R> key."
  print
  print "        To let the darkness claim your soul,"
  print "                   press <Q> key."

  do ' wait for user to select R or Q
    qq$ = ucase$(inkey$)
    if qq$ = "Q" then end
    if qq$ = "R" then run
  loop
end sub

'====================================================================
' Create the player's character
sub CreateCharacter
  cls
  text MM.HRES/2,  0, "Welcome, foolish mortal to the world of", CT
  text MM.HRES/2, 13, "Akalabeth!", CT,1, , rgb(yellow)
  text MM.HRES/2, 26, "Here thou shalf find grand adventure!", CT

  ' Get a number to seed the pseudo random number generator
  ' If zero is entered, use the hardware generator to seed.
  text MM.HRES/2, 45, "Type thy lucky number..", CT
  print @(MM.HRES/2+72,45) "";
  input "",qq$
  luckynum = abs(val(qq$))
  if luckynum = 0 then luckynum = int(rnd * 1234567)  ' use hardware RND for seed
  ' ought to set a maximum value to prevent errors
  zz = rand(-luckynum)   ' seed the PRNG

  text MM.HRES/2, 60, "Choose level of play...", CT
  text MM.HRES/2, 72, "Type a number from 1 to 0 (0 = level 10)",CT,,,rgb(lightgray)
  
  skill = val(WaitUserKey("1234567890"))
  if skill = 0 then skill = 10    ' "0" means level 10
  box 38,71,242,10,1,0,0  ' erase level hint text
  text MM.HRES/2+72, 60, str$(skill), LT

  restore StatNames
  for x = 0 to  5 : read cn$(x) : next  ' names of statistics
  for x = 1 to 10 : read  m$(x) : next  ' names of monsters
  for x = 0 to  5 : read  w$(x) : next  ' names of items
  
  do
    for x = 0 to 5
      c(x)=int(sqr(rand(1))*21+4)
      text 85,  84 + x * 12, cn$(x) + str$(c(x)) + "  ", LT
    next x
    text MM.HRES/2, 160, "Shalt thou play with these qualities? (Y/N)",CT,7
    answer$ = WaitUserKey("YN")
    
  loop until answer$ = "Y"
  box 25,159,270,10,1,0,0   ' erase qualities question text

  text MM.HRES/2, 160, "Art thou a (F)ighter or a (M)age?",CT,7
  pt$ = WaitUserKey("FM")
  box 25,159,270,10,1,0,0   ' erase player type question text

  DisplayPlayerInfo
  GoShopping
end sub

'====================================================================
' Allow player to buy swag
' Use lines 80 to 110 on screen
sub GoShopping
  text  10, 80,"Welcome to the adventure shop."
  text  10, 90,"Which item shalt thou buy?"
  text 175, 90,"(F/R/A/S/B/M/Quit)"
  DisplayPrices
  
  do    ' store purchasing loop
    timer = 0
    do  ' loop until we get any key press
      qq$ = ucase$(inkey$)
      if timer > 2000 then
        text  10,100,space$(50)   ' erase previous purchase after a while
      endif
    loop until qq$ <> ""
    z = -1
    select case qq$
      case "Q","X"  ' Quit or eXit the store
        text 10,100,"Bye.    " + space$(42)
        exit do   ' drop out of store purchasing loop
      case "F"
        text 10,100,"Food.   " + space$(42)
        z=0 : p=1
      case "R"
        text 10,100,"Rapier. " + space$(42)
        if pt$="M" then
          text  60,100,"I'm sorry, mages can't use that!"
          continue do
        else
          z=1 : p=8
        endif
      case "A"
        text 10,100,"Axe.    " + space$(42)
        z=2 : p=5
      case "S"
        text 10,100,"Shield. " + space$(42)
        z=3 : p=6
      case "B"
        text 10,100,"Bow and arrows. " + space$(32)
        if pt$="M" then
          text 110,100,"I'm sorry, mages can't use that!"
          continue do
        else
          z=1 : p=8
        endif
      case "M"
        text 10,100,"Magic Amulet." + space$(40)
        z=5 : p=15
      case else  ' nothing valid selected
        text  10,100,"I'm sorry, we don't have that."
        continue do
    end select

    if c(5) - p < 0 then
      text  10,100,"Thou can not afford it."
      z = -1
    else
      if z = 0 then inc pw(z), 9  ' extra 9 food
      inc pw(z)     ' one more of item
      inc c(5), -p  ' gold has been spent
      text 195,10+z*8,"  " + str$(pw(z)), RT ' update inventory
      text 102,50,str$(c(5)) + "  ", LT  ' update gold
    endif
  loop

  box   0,MM.VRES-32,MM.HRES,32,1,0,0  ' erase text region
  redraw = 1

end sub

'====================================================================
' To allow consistancy, a pseudo-random number generator is employed.
' This allows games to be repeated. It is also key to ensuring dungeon
' level floor plans don't change during the game.
' Based on example routine in the CMM2 User Manual
' parameter p:  negative = use absolute value as new seed value
'               zero = return previous random number
'               positive = ignore value, generate next 'random' number
function rand(p as integer) as float
  static a% = 1103515245, c%=12345, m%=2^31
  static seed% = 7
  
  if p < 0 then
    seed% = abs(p)    ' set new seed
  elseif p > 0 then
    seed%=(a% * seed% + c%) mod m%  ' calc seed next in sequence
  endif
  rand = seed% / m%   ' return random based on current seed
end function

'====================================================================
' Create a dungeon
sub CreateDungeon
  zz = rand(-abs(luckynum)-tx*10-ty*1000+dngLevel*31.4)    ' seed the PRNG
  math set 1,dng%()
  for x = 0 to 4
    for y = 0 to 4
      dng%(2*x+1,2*y+1) = 0
    next y
  next x
  
  for x=2 to 8 step 2
    for y=1 to 9 step 2
      if rand(1)>.95 then dng%(x,y)=2   ' hidden trap
      if rand(1)>.95 then dng%(y,x)=2   ' hidden trap
      if rand(1)>.60 then dng%(y,x)=3   ' hidden door?
      if rand(1)>.60 then dng%(x,y)=3   ' hidden door?
      if rand(1)>.60 then dng%(x,y)=4   ' doorway
      if rand(1)>.60 then dng%(y,x)=4   ' doorway
      if rand(1)>.97 then dng%(y,x)=9   ' trap door down?
      if rand(1)>.97 then dng%(x,y)=9   ' trap door down?
      if rand(1)>.94 then dng%(x,y)=5   ' chest
      if rand(1)>.94 then dng%(y,x)=5   ' chest
      '                   dng%(y,x)=7   ' ladder down
      '                   dng%(y,x)=8   ' ladder up
    next y
  next x

  dng%(2,1)=0 ' ensure space - mainly for level 1 where player first enters
  
  ' add ladders to dungeon level
  if dngLevel/2 = int(dngLevel/2) then
    dng%(7,3) = 7   ' up
    dng%(3,7) = 8   ' dn
  else
    dng%(7,3) = 8
    dng%(3,7) = 7
  endif
  if dngLevel = 1 then dng%(1,1)=8 : dng%(7,3) = 0

  ' place monsters in the dungeon
   ' higher level monsters only appear deeper in dungeon
  nm = 0
  for x = 1 to 10
    mz%(x,0) = 0  ' clear this monster to begin
    'mz%(x,1) = x + 3 + dngLevel ' not needed
    if x - 2 > dngLevel or rand(1) > .4 then continue for
    
    do  ' find a spot to place monster
      ml%(x,0) = int ( rand(1) * 9 + 1)
      ml%(x,1) = int ( rand(1) * 9 + 1)
      if dng%(ml%(x,0),ml%(x,1)) = 0 then exit do   ' place monster in open space
      if ml%(x,0) <> px or ml%(x,1) <> py then exit do  ' no monster on top of player
    loop
  
    dng%(ml%(x,0),ml%(x,1)) = x * 10  ' insert monster
    mz%(x,0) = 1  ' activate monster
    mz%(x,1) = x * 2 + dngLevel * 2 * skill ' monster strength depends on depth and player skill
    nm = nm + 1   ' number of monsters - unused
  
  next x
  redraw = 1

end sub

'====================================================================
' Draw FPV of dungeon
' Drawing is performed from near to far. Variable 'dist' starts at 0 and
' keeps increasing until a wall or door blocks the view.
' A drawback of this method is that distant items over-write nearer ones.
sub DrawDungeonView

  box 0,0,MM.HRES,MM.VRES-32,1,0,0  ' erase graphics region
  dist = 0
  done = 0

  do
    cent = dng%(px+pdx*dist,py+pdy*dist)
    leftside = dng%(px+pdx*dist+pdy,py+pdy*dist-pdx)
    rightside = dng%(px+pdx*dist-pdy,py+pdy*dist+pdx)
    l1 = per%(dist,0) : l2 = per%(dist+1,0)
    r1 = per%(dist,1) : r2 = per%(dist+1,1)
    t1 = per%(dist,2) : t2 = per%(dist+1,2)
    b1 = per%(dist,3) : b2 = per%(dist+1,3)
    cent = int(cent)
    leftside = int(leftside)
    rightside = int(rightside)
    mc = int(cent/10)   ' load possible monster number
    cent = cent-mc*10
    leftside = int((leftside/10-int(leftside/10))*10+.1)
    rightside = int((rightside/10-int(rightside/10))*10+.1)

    if dist <> 0 then
      if cent=1 or cent=3 or cent=4 then
        DrawWallCent
        if cent=4 then DrawDoorCent
        done = 1
      endif
    endif
    
    if not done then
      if leftside=1 or leftside=3 or leftside=4 then DrawWallLeft
      if rightside=1 or rightside=3 or rightside=4 then DrawWallRight
      if leftside=4 and dist>0 then DrawDoorLeft
      if leftside=4 and dist=0 then DrawDoorLeft2
      if rightside=4 and dist>0 then DrawDoorRight
      if rightside=4 and dist=0 then DrawDoorRight2
      
      if leftside<>1 and leftside<>3 and leftside<>4 then
        if dist<>0 then line l1,t1,l1,b1,,RGB(WHITE)
        line l1,t2,l2,t2,,RGB(WHITE)
        line l2,t2,l2,b2,,RGB(WHITE)
        line l2,b2,l1,b2,,RGB(WHITE)
      endif

      if rightside<>1 and rightside<>3 and rightside<>4 then
        if dist<>0 then line r1,t1,r1,b1,,RGB(WHITE)
        line r1,t2,r2,t2,,RGB(WHITE)
        line r2,t2,r2,b2,,RGB(WHITE)
        line r2,b2,r1,b2,,RGB(WHITE)
      endif

      if cent=7 or cent=9 then DrawTrapFloor
      if cent=8 then DrawTrapCeiling
      if cent=7 or cent=8 then DrawLadder

      if cent=5 and dist>0 then
        DrawMimic(dist) 'DrawChest - nah, use mimic routine instead
        ScrollCommands 10, "A chest!"
      endif
    endif

    ' draw possible monster
    if mc > 0 then
      if mc = 8 then
        ScrollCommands 10, "A chest!"   ' a mimic, actually
      else
        ScrollCommands 10, m$(mc)+"!"
      endif
      if dist>0 then    ' draw monster if not on top of player
        select case mc
          case  1 : DrawSkeleton(dist)
          case  2 : DrawThief(dist)
          case  3 : DrawGiantRat(dist)
          case  4 : DrawOrc(dist)
          case  5 : DrawViper(dist)
          case  6 : DrawCarrionCrawler(dist)
          case  7 : DrawGremlin(dist)
          case  8 : DrawMimic(dist)
          case  9 : DrawDaemon(dist)
          case 10 : DrawBalrog(dist)
        end select
      endif
    endif
    dist = dist + 1   ' another step farther away
  loop until done
  
  if debugview then
    DrawDungeonPlanView
  endif

end sub

' FPV support routines
' draw wall
sub DrawWallCent
  line l1,t1,r1,t1,,RGB(WHITE)
  line r1,t1,r1,b1,,RGB(WHITE)
  line r1,b1,l1,b1,,RGB(WHITE)
  line l1,b1,l1,t1,,RGB(WHITE)
end sub

' draw door in front
sub DrawDoorCent
  line cd%(dist,0),cd%(dist,3),cd%(dist,0),cd%(dist,2),,RGB(WHITE) 
  line cd%(dist,0),cd%(dist,2),cd%(dist,1),cd%(dist,2),,RGB(WHITE)
  line cd%(dist,1),cd%(dist,2),cd%(dist,1),cd%(dist,3),,RGB(WHITE)
end sub

' draw left side wall
sub DrawWallLeft
  line l1,t1,l2,t2,,RGB(WHITE)
  line l1,b1,l2,b2,,RGB(WHITE)
end sub

' draw right side wall
sub DrawWallRight
  line r1,t1,r2,t2,,RGB(WHITE)
  line r1,b1,r2,b2,,RGB(WHITE)
end sub

' draw left side door
sub DrawDoorLeft
  line ld%(dist,0),ld%(dist,4),ld%(dist,0),ld%(dist,2),,RGB(WHITE)
  line ld%(dist,0),ld%(dist,2),ld%(dist,1),ld%(dist,3),,RGB(WHITE)
  line ld%(dist,1),ld%(dist,3),ld%(dist,1),ld%(dist,5),,RGB(WHITE)
end sub

' draw left side half-door
sub DrawDoorLeft2
  line 0,ld%(dist,2)-3,ld%(dist,1),ld%(dist,3),,RGB(WHITE)
  line ld%(dist,1),ld%(dist,3),ld%(dist,1),ld%(dist,5),,RGB(WHITE)
end sub

' draw right side door
sub DrawDoorRight
  line 319-ld%(dist,0),ld%(dist,4),319-ld%(dist,0),ld%(dist,2),,RGB(WHITE)
  line 319-ld%(dist,0),ld%(dist,2),319-ld%(dist,1),ld%(dist,3),,RGB(WHITE) 
  line 319-ld%(dist,1),ld%(dist,3),319-ld%(dist,1),ld%(dist,5),,RGB(WHITE)
end sub

' draw right side half-door
sub DrawDoorRight2
  line 319,ld%(dist,2) - 3,319 - ld%(dist,1),ld%(dist,3),,RGB(WHITE)
  line 319 - ld%(dist,1),ld%(dist,3),319 - ld%(dist,1),ld%(dist,5),,RGB(WHITE)
end sub ' 32203 return

' draw trap door on floor
sub DrawTrapFloor
  line ft%(dist,0),ft%(dist,4),ft%(dist,2),ft%(dist,5),,RGB(WHITE)
  line ft%(dist,2),ft%(dist,5),ft%(dist,3),ft%(dist,5),,RGB(WHITE)
  line ft%(dist,3),ft%(dist,5),ft%(dist,1),ft%(dist,4),,RGB(WHITE)
  line ft%(dist,1),ft%(dist,4),ft%(dist,0),ft%(dist,4),,RGB(WHITE)
end sub

' draw trap door on ceiling
sub DrawTrapCeiling
  line ft%(dist,0),158-ft%(dist,4),ft%(dist,2),158-ft%(dist,5),,RGB(WHITE)
  line ft%(dist,2),158-ft%(dist,5),ft%(dist,3),158-ft%(dist,5),,RGB(WHITE)
  line ft%(dist,3),158-ft%(dist,5),ft%(dist,1),158-ft%(dist,4),,RGB(WHITE)
  line ft%(dist,1),158-ft%(dist,4),ft%(dist,0),158-ft%(dist,4),,RGB(WHITE)
end sub

sub DrawLadder
  local base,top,xleft,xright,y1,y2,y3,y4
  
  base=ladder%(dist,3)
  top=ladder%(dist,2)
  xleft=ladder%(dist,0)
  xright=ladder%(dist,1)
  y1=(base*4+top)/5
  y2=(base*3+top*2)/5
  y3=(base*2+top*3)/5
  y4=(base+top*4)/5
  line xleft,base,xleft,top,2,   RGB(BROWN)   ' thicker lines for poles
  line xright,base,xright,top,2, RGB(BROWN)
  line xleft,y1,xright,y1,2,     RGB(BROWN)   ' thicker lines for rungs
  line xleft,y2,xright,y2,2,     RGB(BROWN)
  line xleft,y3,xright,y3,2,     RGB(BROWN)
  line xleft,y4,xright,y4,2,     RGB(BROWN)
end sub

' Draw top view of dungeon, for debugging (& cheating)
sub DrawDungeonPlanView
  local i,j,monst
  box 135,0,MM.HRES-135,MM.VRES-32,1,0,0  ' erase graphics region
  text 49, 37,chr$(133),RM,1: text 51, 37," Door",LM    ' show symbol key
  text 49, 50,chr$(134),RM,1: text 51, 50," Hidden Door",LM
  text 49, 63,chr$(130),RM,1: text 51, 63," Hidden Trap",LM
  text 49, 76,chr$(147),RM,1: text 51, 76," Ladder Down",LM
  text 49, 89,chr$(146),RM,1: text 51, 89," Ladder Up",LM
  text 49,102,chr$(151),RM,1: text 51,102," Trap Door",LM
  text 49,115,chr$(166),RM,1,,RGB(YELLOW): text 51,115," Chest",LM,,,RGB(YELLOW)
  text 71,135,"Forward: ",RM
  if pdx then text 73,135,chr$(148.5-pdx/2),LM,1
  if pdy then text 73,135,chr$(146.5+pdy/2),LM,1
  for i = 0 to 10   ' draw maze
    for j = 0 to 10
      select case dng%(i,j)
        case 1   ' wall
          box 135+i*15,1+j*15,15,15,1,RGB(GREY),RGB(GREY)
        case 2   ' trap
          text 142+i*15,8+j*15,chr$(130),CM,1
        case 3   ' hidden door
          text 142+i*15,8+j*15,chr$(134),CM,1
        case 4   ' door
          text 142+i*15,8+j*15,chr$(133),CM,1
        case 5   ' chest
          text 142+i*15,8+j*15,chr$(166),CM,1,,RGB(YELLOW)
        case 7   ' ladder down
          text 142+i*15,8+j*15,chr$(147),CM,1
        case 8   ' ladder up
          text 142+i*15,8+j*15,chr$(146),CM,1
        case 9   ' trap door down
          text 142+i*15,8+j*15,chr$(151),CM,1
        case else
          monst = int(dng%(i,j)/10)
          if monst >= 1 then
            text 142+i*15,9+j*15,chr$(47+monst),CM,1,,RGB(RED)
          endif
      end select
    next j
  next i
  text 142+px*15,9+py*15,chr$(152),CM,1,,RGB(GREEN),-1 ' show player position
end sub

'====================================================================
' Handle each possible monster on level - there could be one of each.
sub MonsterHandle
  local n%
  for n% = 1 to 10
    if mz%(n%,0) then MonsterAction(n%,ML%(n%,0),ML%(n%,1))
  next n%
end sub

' Let monster n, located at mx,my, do it's thing.
' Weak monsters will try to run away, unless cornered.
' Distant strong monsters will try to move closer to the player.
' A monster up close to (beside) the player will attack.
' A thief or a gremlin might steal something instead of attacking.
' Mimics won't move closer if near (within 3 of) the player - sneaky critters!
' note: mx & my are passed by reference so that changes are returned
sub MonsterAction(n,mx as integer,my as integer)
  local integer mdx,mdy
  local float ra   ' distance from monster to player in dungeon

  ra = sqr((px - mx) ^ 2 + (py - my) ^ 2)  ' distance to player
  mdx = sgn(px - mx)   ' direction toward player
  mdy = sgn(py - my)   '    -1/0/1 for x and y axes
  
  if mz%(n,1) < dngLevel*skill then  ' all weak monsters try to run away
    mdx = -mdx  ' change direction to
    mdy = -mdy  '    away from player
    MonsterMove(n,mx,my,mdx,mdy,ra)
  elseif ra < 1.3 then  ' attack if beside player
    MonsterAttack(n)
  elseif n = 8 and ra < 3 then
    exit sub  ' strong mimic doesn't move closer if within 3
  else  ' far away from player so try to move closer
    MonsterMove(n,mx,my,mdx,mdy,ra)
  endif
end sub

' Monster n at mx,my will try to move in direction mdy, then try mdx.
' A weak monster, if unable to move away, will attack if close - ie cornered
' note: mx & my are passed by reference so that changes are returned
sub MonsterMove(n,mx as integer,my as integer,mdx,mdy,ra)
  local integer d
  if mdy <> 0 then   ' no y-move so check for x-move
    d = dng%(mx,(my + mdy + .005))   ' What's in the y direction?
    if d = 1 or d = 2 or d > 9 then
      mdy = 0   ' can't move that way because wall, trap or another monster is there
    else
      mdx = 0   ' perform a y-move only
    endif
  endif
  
  if mdx <> 0 then  ' no y-move so check for x-move
    d = dng%((mx + mdx + .005),my)   ' What's in the x direction?
    if d = 1 or d = 2 or d > 9 then
      mdx = 0  ' can't move that way because wall, trap or another monster is there
    endif
  endif

  if mdx = 0 and mdy = 0 then ' monster couldn't move
    if mz%(n,1) < dngLevel*skill then   ' weak monster couldn't run
      if ra < 1.3 then  ' desparation attack if beside player
        MonsterAttack(n)
      else
        mz%(n,1) = mz%(n,1) + n + dngLevel   ' heal monster somewhat
      endif
    endif
  else  ' movin, movin, movin, let's get that monster movin
    if mx + mdx = px and my + mdy = py then exit sub  ' on top of player? should never be true
    dng%(mx,my) = dng%(mx,my) - 10 * n '  erase monster at old location
    mx = mx + mdx:my = my + mdy   ' passed by reference, must propagate back
    dng%(mx,my) = (dng%(mx,my) + 10 * n + .005)  ' replace monster at new
    redraw = 1
  endif
end sub

' Monster attacks!
' Unless it's a thief or gremlin, in which case there is
' a 50% chance it will steal something instead
sub MonsterAttack(n)
  local integer t
  if n = 2 and rand(1) < .5 then ' yes, it's stealin thief time
    do
      t = int(rand(1) * 6)
    loop until pw(t) > 0    ' find something to steal
    ScrollCommands 10, "A thief stole a "+w$(t)
    pw(t) = pw(t) - 1
    'if pa = 1 then print "-CR- to cont. ";: input q$   ' pause mode
    exit sub
  endif
      
  if n = 7 and rand(1) < .5 then  ' yes, it's food stealin gremlin time
    ScrollCommands 10, "A gremlin stole some food."
    pw(0) = int(pw(0) / 2)
    'if pa = 1 then print "-CR- to cont. ";: input q$   ' pause mode
    exit sub
  endif

  ScrollCommands  1, "You are being attacked by a "+m$(n)
  if rand(1) * 20 - sgn(pw(3)) - c(3) + n + dngLevel < 0 then
    ScrollCommands 20, "Missed."
  else
    ScrollCommands 25, "Hit!"
    c(0) = max(0, c(0) - int(rand(1) * n + dngLevel))
  endif
  'if pa = 1 then print "-CR- to cont. ";: input q$   ' pause mode
end sub

'====================================================================
' Draw monster at data pointer at requested distance
' Bottom center of monster is in xx%(0),yy%(0)
sub DrawMonster(dist)
  local mtry = yy%(0) + yy%(dist)
  local xb,xe,yb,ye

  do
    read col  ' read colour of next polyline
    if col = 0 then exit do   ' end of image
    read xb,yb  ' read start vertex
    do
      read xe,ye  ' read next vertex
      if xe = 0 and ye = 0 then exit do   ' end of polyline
      line xx%(0)+xb/dist,mtry-yb/dist , xx%(0)+xe/dist,mtry-ye/dist ,,col
      xb = xe
      yb = ye
    loop
  loop
end sub

sub DrawSkeleton(dist)
  restore DatSkeleton
  DrawMonster(dist)
end sub

sub DrawThief(dist)
  restore DatThief
  DrawMonster(dist)
end sub

sub DrawGiantRat(dist)
  restore DatGiantRat
  DrawMonster(dist)
end sub

sub DrawOrc(dist)
  restore DatOrc
  DrawMonster(dist)
end sub

sub DrawViper(dist)
  restore DatViper
  DrawMonster(dist)
end sub

' not encoded yet because there are strange y references - the ceiling perhaps
sub DrawCarrionCrawler(dist)
  local mtry = yy%(0) + yy%(dist)

  line xx%(0)+20/dist,mtry-88/dist,xx%(0)+20/dist,yy%(0)-yy%(dist),,RGB(MAGENTA)
  line xx%(0)+20/dist, yy%(0)-yy%(dist),xx%(0)-20/dist,yy%(0)-yy%(dist),,RGB(MAGENTA)
  line xx%(0)-20/dist, yy%(0)-yy%(dist),xx%(0)-20/dist,mtry-88/dist,,RGB(MAGENTA)
  
  line xx%(0)-20/dist,mtry-88/dist,xx%(0)-10/dist,mtry-83/dist,,RGB(MAGENTA)
  line xx%(0)-10/dist,mtry-83/dist,xx%(0)+10/dist,mtry-83/dist,,RGB(MAGENTA)
  line xx%(0)+10/dist,mtry-83/dist,xx%(0)+20/dist,mtry-88/dist,,RGB(MAGENTA)
  line xx%(0)-20/dist,mtry-88/dist,xx%(0)-30/dist,mtry-83/dist,,RGB(MAGENTA)
  line xx%(0)-30/dist,mtry-83/dist,xx%(0)-30/dist,mtry-78/dist,,RGB(MAGENTA)
  line xx%(0)+20/dist,mtry-88/dist,xx%(0)+30/dist,mtry-83/dist,,RGB(MAGENTA)
  line xx%(0)+30/dist,mtry-83/dist,xx%(0)+40/dist,mtry-83/dist,,RGB(MAGENTA)
  line xx%(0)-15/dist,mtry-86/dist,xx%(0)-20/dist,mtry-83/dist,,RGB(MAGENTA)
  line xx%(0)-20/dist,mtry-83/dist,xx%(0)-20/dist,mtry-78/dist,,RGB(MAGENTA)
  line xx%(0)-20/dist,mtry-78/dist,xx%(0)-30/dist,mtry-73/dist,,RGB(MAGENTA)
  line xx%(0)-30/dist,mtry-73/dist,xx%(0)-30/dist,mtry-68/dist,,RGB(MAGENTA)
  line xx%(0)-30/dist,mtry-68/dist,xx%(0)-20/dist,mtry-63/dist,,RGB(MAGENTA)
  line xx%(0)-10/dist,mtry-83/dist,xx%(0)-10/dist,mtry-58/dist,,RGB(MAGENTA)
  line xx%(0)-10/dist,mtry-58/dist,xx%(0)        ,mtry-50/dist,,RGB(MAGENTA)
  line xx%(0)+10/dist,mtry-83/dist,xx%(0)+10/dist,mtry-78/dist,,RGB(MAGENTA)
  line xx%(0)+10/dist,mtry-78/dist,xx%(0)+20/dist,mtry-73/dist,,RGB(MAGENTA)
  line xx%(0)+20/dist,mtry-73/dist,xx%(0)+20/dist,mtry-40/dist,,RGB(MAGENTA)
  line xx%(0)+15/dist,mtry-85/dist,xx%(0)+20/dist,mtry-78/dist,,RGB(MAGENTA)
  line xx%(0)+20/dist,mtry-78/dist,xx%(0)+30/dist,mtry-76/dist,,RGB(MAGENTA)
  line xx%(0)+30/dist,mtry-76/dist,xx%(0)+30/dist,mtry-60/dist,,RGB(MAGENTA)
  line xx%(0)        ,mtry-83/dist,xx%(0)        ,mtry-73/dist,,RGB(MAGENTA)
  line xx%(0)        ,mtry-73/dist,xx%(0)+10/dist,mtry-68/dist,,RGB(MAGENTA)
  line xx%(0)+10/dist,mtry-68/dist,xx%(0)+10/dist,mtry-63/dist,,RGB(MAGENTA)
  line xx%(0)+10/dist,mtry-63/dist,xx%(0)        ,mtry-58/dist,,RGB(MAGENTA)
end sub

sub DrawGremlin(dist)
  restore DatGremlin
  DrawMonster(dist)
end sub

sub DrawMimic(dist)
  restore DatMimic
  DrawMonster(dist)
end sub

sub DrawDaemon(dist)
  restore DatDaemon
  DrawMonster(dist)
end sub

sub DrawBalrog(dist)
  restore DatBalrog
  DrawMonster(dist)
end sub

' Various text strings
StatNames:
  data "Hit points.....","Strength.......","Dexterity......"
  data "Stamina........","Wisdom.........","Gold..........."

  data "Skeleton","Thief","Giant Rat","Orc","Viper"
  data "Carrion Crawler","Gremlin","Mimic","Daemon","Balrog"

  data "Food","Rapier","Axe","Shield","Bow and arrows","Magic amulet"

' Scaleable Vector Graphics for each monster
' Data format: images are stored as one or more coloured polylines
' colour,[x,y] repeated for each vertex, 0,0 to end this polyline
' 0,0,0 inidicates end of image
' Coordinate 0,0 is bottom center of image, -x is left, +y is up.
' Note no vertex can be placed at 0,0
DatSkeleton:
  data RGB(WHITE),-23,0,-15,0,-15,15,-8,30,8,30,15,15,15,0,23,0,0,0
  data RGB(WHITE),0,26,0,65,0,0
  data RGB(WHITE),-2,38,2,38,0,0
  data RGB(WHITE),-3,45,3,45,0,0
  data RGB(WHITE),-5,53,5,53,0,0
  data RGB(WHITE),-23,56,-30,53,-23,45,-23,53,-8,38,0,0
  data RGB(WHITE),-15,45,-8,60,8,60,15,45,0,0
  data RGB(WHITE),15,42,15,57,0,0
  data RGB(WHITE),12,45,20,45,0,0
  data RGB(WHITE),0,75,-5,80,-8,75,-5,65,5,65,5,68,-5,68,-5,65,5,65,8,75,5,80,-5,80,0,0
  data RGB(RED),-5,72,-5,72,0,0
  data RGB(RED),5,72,5,72,0,0,0,0,0

DatThief:
  data RGB(BLUE),0,56,0,8,10,0,30,0,30,45,10,64,0,56,-10,64,-30,45,-30,0,-10,0,0,8,0,0
  data RGB(BLUE),-10,64,-10,75,0,83,10,75,0,79,0,0
  data RGB(BLUE),-10,75,0,79,-10,75,0,60,10,75,10,64,0,0,0,0,0

DatGiantRat:
  data RGB(GRAY),5,30,0,25,-5,30,-15,5,-10,0,10,0,15,5
  data 20,5,10,0,15,5,5,30,10,40,3,35,-3,35,-10,40,-5,30,0,0
  data RGB(GRAY),-5,20,-5,15,0,0
  data RGB(GRAY),5,20,5,15,0,0
  data RGB(GRAY),-7,20,-7,15,0,0
  data RGB(GRAY),7,20,7,15,0,0
  data RGB(RED),-5,33,-2,30,0,0
  data RGB(RED),5,33,2,30,0,0,0,0,0

DatOrc:
  data RGB(BROWN),-15,0,-8,8,-8,15,-15,23,-15,15,-23,23,-23,45,-15,53,-8,53,-15,68
  data -8,75,8,75,15,68,8,53,15,53,23,45,23,23,15,15,15,23,8,15,8,8,15,0,-15,0,0,0
  data RGB(BROWN),-15,68,15,68,0,0
  data RGB(BROWN),-8,53,8,53,0,0
  data RGB(BROWN),-23,15,8,45,0,0
  data RGB(BROWN),-8,68,0,60,8,68,8,60,-8,60,-8,68,0,0
  data RGB(BROWN),0,38,-8,38,8,53,8,45,15,45,0,30,0,38,0,0,0,0,0

DatViper:
  data RGB(RED),-7,43,-3,44,0,0
  data RGB(RED),7,43,3,44,0,0
  data RGB(GREEN),-10,15,-10,30,-15,20,-15,15,-15,0,15,0,15,15,-15,15,0,0
  data RGB(GREEN),-15,10,15,10,0,0
  data RGB(GREEN),-15,5,15,5,0,0
  data RGB(GREEN),0,15,-5,20,-5,35,5,35,0,0
  data RGB(GREEN),5,35,5,20,10,15,0,0
  data RGB(GREEN),-5,20,5,20,0,0
  data RGB(GREEN),-5,25,5,25,0,0
  data RGB(GREEN),-5,30,5,30,0,0
  data RGB(GREEN),-10,35,-10,40,-5,45,5,45,10,40,10,35,0,0
  data RGB(GREEN),-10,40,0,45,10,40,0,0
  data RGB(GREEN),-5,40,5,40,15,30,0,40,-15,30,-5,40,0,0,0,0,0

DatCarrionCrawler:

DatGremlin:
  data RGB(ORANGE),5,10,-5,10,0,15,10,20,5,15,5,10,7,6,5,3,-5,3,-7,6,-5,10,0,0
  data RGB(ORANGE),2,3,5,0,8,0,0,0
  data RGB(ORANGE),-2,3,-5,0,-8,0,0,0
  data RGB(ORANGE),3,5,-3,5,0,0
  data RGB(RED),3,8,3,8,0,0
  data RGB(RED),-3,8,-3,8,0,0,0,0,0

DatMimic:
  data RGB(BROWN),-10,0,-10,10,10,10,10,0,-10,0,0,0
  data RGB(BROWN),-10,10,-5,15,15,15,15,5,10,0,0,0
  data RGB(BROWN),10,10,15,15,0,0,0,0,0

DatDaemon:
  data RGB(RED),-14,46,-12,37,-20,32,-30,32,-22,24,-40,17,-40,7,-38,5,-40,3,-40,0
  data -36,0,-34,2,-32,0,-28,0,-28,3,-30,5,-28,7,-28,15,0,27,0,0
  data RGB(RED),14,46,12,37,20,32,30,32,22,24,40,17,40,7,38,5,40,3,40,0,36,0,34,2
  data 32,0,28,0,28,3,30,5,28,7,28,15,0,27,0,0
  data RGB(RED),6,48,38,41,40,42,18,56,12,56,10,57,8,56,-8,56,-10,58,14,58,16,59,8,63
  data 6,63,2,70,2,63,-2,63,-2,70,-6,63,-8,63,-16,59,-14,58,-10,57,-12,56,-18,56,-36,47
  data -36,39,-28,41,-28,46,-20,50,-18,50,-14,46,0,0
  data RGB(YELLOW),-28,41,30,55,0,0
  data RGB(YELLOW),28,58,22,56,22,53,28,52,34,54,0,0
  data RGB(RED),20,50,26,47,0,0
  data RGB(RED),10,58,10,61,4,58,0,0
  data RGB(RED),-10,58,-10,61,-4,58,0,0
  data RGB(RED),40,9,50,12,40,7,0,0
  data RGB(RED),-8,25,6,7,6,7,28,7,28,7,28,9,28,9,20,9,20,9,6,25,0,0,0,0,0

DatBalrog:
  data RGB(GOLD),6,60,30,90,60,30,60,10,30,40,15,40,0,0
  data RGB(GOLD),-6,60,-30,90,-60,30,-60,10,-30,40,-15,40,0,0
  data RGB(GOLD),0,25,6,25,10,20,12,10,10,6,10,0,14,0,15,5
  data 16,0,20,0,20,6,18,10,18,20,15,30,15,45,40,60,40,70
  data 10,55,6,60,10,74,6,80,4,80,3,82,2,80,0,80,0,0
  data RGB(GOLD),0,25,-6,25,-10,20,-12,10,-10,6,-10,0,-14,0,-15,5
  data -16,0,-20,0,-20,6,-18,10,-18,20,-15,30,-15,45,-40,60,-40,70
  data -10,55,-6,60,-10,74,-6,80,-4,80,-3,82,-2,80,0,80,0,0
  data RGB(GOLD),-6,25,0,6,10,0,4,8,6,25,0,0
  data RGB(GOLD),-40,64,-40,90,-52,80,-52,40,0,0
  data RGB(GOLD),40,86,38,92,42,92,40,86,40,50,0,0
  data RGB(GOLD),4,70,6,74,0,0
  data RGB(GOLD),-4,70,-6,74,0,0
  data RGB(GOLD),0,64,0,60,0,0,0,0,0

  
  