
' Falfus2
' for the Colour Maximite 2
' written by vegipete, 2020
'
' Slide and drop identical tiles together to make them
' vanish. Clear all tiles to advance to the next level.
' Just don't leave any single tiles...
'
' Works with a Wii Nunchuk on I2C3 (front connector)
' Uses the keyboard if no Nunchuk found
'   (arrow keys to move, [a] to grab and release)
'
' Press [space] to restart level,
'       [c] to enter level code,
'       [s] to save current image (optional - normally REM'ed out)
'
' Files to be paced in the same directory:
'   Falfus2XXXX             (this file)
'   tiles30x30x12x2.bmp     (sprites and images)
'   Slide.wav               (various sound effects)
'   Clink.wav
'   Pop.wav
'
'
' Version history
' Falfus      written decades ago for the Apple//gs
' Falfus2br1  first release for the CMM2
' Falfus2br2  added keyboard support
' Falfus2-r3  added level codes
'
' Falfus is loosely based on the game Puznik.

cls

'mode 2,8  ' 640x400x8 bit
mode 1,8

CONST XOFFSET = 40
CONST YOFFSET = 100
CONST SPRAYSIZE = 2
CONST PLAYWIDTH = 16
CONST PLAYHEIGHT = 12   ' 13 tiles high  (remember array index 0)

CONST MAXLEVEL = 60   ' restore command only works with literals. GRRRRR....

dim string levelsource(MAXLEVEL,PLAYHEIGHT) length PLAYWIDTH+1
dim string currentlevel(PLAYHEIGHT) length PLAYWIDTH+1
dim string fusem(PLAYHEIGHT) length PLAYWIDTH+1  ' array of tiles to explode
dim integer htile, htileleft, htileright
dim integer hx,hy,cx,cy,action_button

cx = PLAYWIDTH/2
cy = PLAYHEIGHT/2
hx = (cx + 7) * 30 + XOFFSET
hy = cy * 30 + YOFFSET
action_button = 0

' load tile images onto page 2
page write 2 : cls
load bmp "tiles30x30x12x2.bmp"

sprite read 1,300,0,30,30 ' sprite-ify the open hand
sprite read 2,330,0,30,30 ' sprite-ify the closed hand

' test for nunchuk
GotNun = 1
on error skip
controller nunchuk open 3
if MM.ERRNO = 16 then
  GotNun = 0
  page write 0
  text MM.HRES/2,100,"No Nunchuk found.","CT",1,2,rgb(yellow)
  tt$="Use "+chr$(149)+chr$(146)+chr$(147)+chr$(148)+" to move"
  text MM.HRES/2,150,tt$,"CT",1,2,rgb(yellow)
  text MM.HRES/2,200,"and [A] to grab and release.","CT",1,2,rgb(yellow)
  text MM.HRES/2,250,"Press any key to continue...","CT",1,1,rgb(cyan)
  do : loop until inkey$ <> ""
endif

' use page 1 as page drawing buffer
page write 1 : cls
sprite show 1,MM.HRES-1,100,0 ' put both offscreen
sprite show 2,MM.HRES-1,200,0

ReadAllLevels

level = 1
tilecount = 1   ' updated by DrawPlayingArea
lastfall = 0    ' keep track of falling tiles

do
  page write 1
  DrawBrickWall   ' fill the screen with bricks
  DrawMetalSign
  page copy 1 to 0, B  ' wait for blanking, then show wall and sign
  d = LoadLevel(level)
  DrawCurrentLevel
  STROKESPEED = 0
  PaintTitle(XOFFSET,YOFFSET,10,rgb(green))

  page write 0
  STROKESPEED = 10
  'pause 1000  ' wait for VGA mode to settle
  PaintTitle(XOFFSET,YOFFSET,10,rgb(green))

  page write 1
  instruct$ = "Press [space bar] to restart level, [c] to enter level code."
  text 500,395+YOFFSET,instruct$, "CT" , 1, 1, &hA0A0A0, -1

  do
    ' read user input
    if GotNun then
      ReadNunchuk
    endif

    ' test action button
    if action_button then  ' is it pressed?
      TryShiftTiles
    else   'no, it's released
      locky = 0
      sprite show 2,MM.HRES-1,200,0  ' put closed hand offscreen
      sprite show 1,hx,hy,0   ' show open hand
    endif

    if tilecount = 0 then
      ShowLevelCode(level)
      level = level + 1
      sprite show 1,MM.HRES-1,100,0 ' put them offscreen
      sprite show 2,MM.HRES-1,200,0
      if not LoadLevel(level) then exit do
      DrawCurrentLevel
    endif

    k$ = inkey$
    if k$ <> "" then
      kp$ = lcase$(kp$)   ' only checking for lower case
      if  k$ = " " then   ' restart level if [SPACE] pressed
        sprite show 1,MM.HRES-1,100,0 ' put them offscreen
        sprite show 2,MM.HRES-1,200,0
        d = LoadLevel(level)   ' reload the level layout
        DrawCurrentLevel
      elseif k$ = chr$(27) then  ' [ESC] key
        print @(0,400)
        end
'      elseif k$ = "s" then save image "FFpic"
      elseif k$ = "c" then
        sprite show 1,MM.HRES-1,100,0 ' move both cursors offscreen
        sprite show 2,MM.HRES-1,200,0
        nlev = SelectLevel()
        if nlev then level = nlev
        d = LoadLevel(level)
        DrawCurrentLevel
      else
        HandleKeyboard(asc(k$))
      endif
    endif

    page copy 1 to 0,B ' display the buffer
    pause 15

  loop

  GameFinished

loop

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Oh drat! We've run out of levels to offer the player.
sub GameFinished

  page write 0
  rbox 300,250,400,100,10,rgb(yellow),rgb(yellow)
  rbox 304,254,392, 92,10,rgb(yellow),rgb(black)
  text 500,290, "CONGRATULATIONS!", "CB" , 3, 1, &hFFB000, -1
  text 500,294, "You have completed all levels.", "CT" , 2, 1, &hFFB000, -1
  text 500,340, "Press the space bar to quit...", "CB" , 1, 1, &hA0A0A0, -1
  ? @(0,350)
  do
  loop until INKEY$ <> ""
  end

end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Let player enter a code
' Return level number if it matches, 0 otherwise
function SelectLevel()
  local kp$,code$

  rbox MM.HRES/2-107,MM.VRES/2-38,300,80,15,&h404040,&h404040
  rbox MM.HRES/2-103,MM.VRES/2-34,291,72,12,&h607080,&h607080
  text MM.HRES/2+43,MM.VRES/2-28,"Enter Level Code:","CT",3,1,&hFFB000,-1
  box MM.HRES/2-42,MM.VRES/2+2,170,25,1,0,0
  code$ = ""
  do
    text MM.HRES/2+43,MM.VRES/2+4," "+code$+" ","CT",2,1,&hC0FF00
    text MM.HRES/2+42,MM.VRES/2+4," "+code$+" ","CT",2,1,&hC0FF00,-1
    text MM.HRES/2+44,MM.VRES/2+4," "+code$+" ","CT",2,1,&hC0FF00,-1
    page copy 1 to 0,B ' display the buffer
    do
      kp$ = inkey$
    loop until kp$ <> ""  ' loop until key press detected
    select case kp$
      case "0" to "9"
        if len(code$) < 10 then code$ = code$ + kp$
      case chr$(8), chr$(127), chr$(130)
        if len(code$) then code$ = left$(code$,len(code$)-1)
      case chr$(27)
        SelectLevel = 0
        exit function
      case chr$(13)
        exit do
    end select
  loop

  for fl = 1 to MAXLEVEL-1
    tc = GenerateLevelCode(fl)
    if format$(tc,"%10.0f") = code$ then
      SelectLevel = fl+1
      exit function
    endif
  next fl

  SelectLevel = 0

end function

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
sub ShowLevelCode(thlev)
  local code$

  rbox MM.HRES/2-112,MM.VRES/2-38,310,80,15,&h404040,&h404040
  rbox MM.HRES/2-108,MM.VRES/2-34,301,72,12,&h607080,&h607080
  text MM.HRES/2+43,MM.VRES/2-28,"Level "+str$(thlev)+" Complete!","CT",3,1,&hFFB000,-1
  text MM.HRES/2+44,MM.VRES/2-28,"Level "+str$(thlev)+" Complete!","CT",3,1,&hFFB000,-1
  cd = GenerateLevelCode(thlev)
  code$ = format$(cd,"%10.0f")
  
  text MM.HRES/2+43,MM.VRES/2+4,"Level code: "+code$,"CT",2,1,&hFFB000,-1
  page copy 1 to 0,B ' display the buffer
  do : loop until inkey$ <> "" or nunchuk(Z,3)
end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
sub ReadNunchuk

    hx = ((nunchuk(JX,3) - 40) * 2.85) + 220 + XOFFSET
    hy = ((220 - nunchuk(JY,3)) * 2.1) +   0 + YOFFSET
    action_button = nunchuk(Z,3)
    
    if hx > 660+XOFFSET then hx = 660+XOFFSET
    if hx < 270+XOFFSET then hx = 270+XOFFSET
    if hy > 330+YOFFSET then hy = 330+YOFFSET
    if hy <  30+YOFFSET then hy =  30+YOFFSET

    ' calculate which cell the hand is in
    cx = int((hx - XOFFSET + 15) / 30) - 7
    cy = int((hy - YOFFSET + 15) / 30)

end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
sub HandleKeyboard(letter)

  select case letter
    case 128
      cy = cy - (cy > 1)
    case 129
      cy = cy + (cy < PLAYHEIGHT-1)
    case 130
      cx = cx - (cx > 2)
    case 131
      cx = cx + (cx < PLAYWIDTH-1)
    case asc("A"),asc("a")
      action_button = (action_button=0)
  end select
  hx = (cx + 7) * 30 + XOFFSET
  hy = cy * 30 + YOFFSET
end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
sub TryShiftTiles

  sprite show 1,MM.HRES-1,100,0 ' put open hand offscreen
  if locky = 0 then ' look for button down event
    locky = cy
    lockx = cx
    htile      = GetTile(lockx  ,locky)
    htileleft  = GetTile(lockx-1,locky)
    htileright = GetTile(lockx+1,locky)
  else  ' button already down so look for side shift
    if htile > asc("0") and htile < asc("9") then
      if cx < lockx and htileleft = asc(".") then ' can we shift left?
        sprite show 2,MM.HRES-1,200,0  ' move closed hand offscreen to restore image
        SetTile(lockx,locky,asc("."))  ' erase tile at old position
        lockx = lockx - 1
        SetTile(lockx,locky,htile) ' move tile to new position
        htileleft  = GetTile(lockx-1,locky)
        htileright = asc(".")   ' leave blank behind us
        ' play scraping tile sound
        play stop : play wav "Slide.wav"
        DrawPlayingArea
        page copy 1 to 0,B ' show new image
'        pause 25
        if DoPhysics() then
          htile=GetTile(lockx,locky) ' may need to let go of the tile
          if htile = asc(".") then action_button = 0  ' tile fell so open hand
        endif
      endif  
      if cx > lockx and htileright = asc(".") then ' can we shift right?
        sprite show 2,MM.HRES-1,200,0  ' move closed hand offscreen to restore image
        SetTile(lockx,locky,asc("."))  ' erase tile at old position
        lockx = lockx + 1
        SetTile(lockx,locky,htile) ' move tile to new position
        htileleft  = asc(".")   ' leave blank behind us
        htileright = GetTile(lockx+1,locky)
        ' play scraping tile sound
        play stop : play wav "Slide.wav"
        DrawPlayingArea
        page copy 1 to 0,B ' show new image
'        pause 25
        if DoPhysics() then 
          htile=GetTile(lockx,locky) ' may need to let go of the tile
          if htile = asc(".") then action_button = 0  ' tile fell so open hand
        endif
      endif
    endif  
  endif
  sprite show 2,hx,locky*30+YOFFSET,0 ' show closed hand locked vertically to tile

end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Check for and move pieces according to gravity and fusion
' Do gravity (falling down) first, from bottom to top until
' nothing more falls.
' Then do fusion.
' Repeat until there are no more changes. (Fusion could open
' new gaps, allowing more tiles to fall.)
function DoPhysics()

  local result,r

  result = 0
  do  ' keep looping until nothing fuses
    lastfall = 0
    do  ' keep looping until nothing falls
      r = DoFalling()
      if r then
        DrawPlayingArea
        page copy 1 to 0,B ' show new image
        pause 100
        result = 1
      else
        ' clink sound only at bottom of falling
        if lastfall then play stop : play wav "Clink.wav"
        exit do
      endif
    loop

    if DoFusing() = 0 then exit do
    result = 1
  loop

  DoPhysics = result

end function

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Perform a single iteration of falling tiles
function DoFalling()

  local integer tc

  changed = 0
  for y = PLAYHEIGHT-2 to 1 step -1
    for x = 2 to PLAYWIDTH-1
      tc = GetTile(x,y)
      if tc > asc("0") and tc < asc("9") then ' is this a playing tile?
        if GetTile(x,y+1) = asc(".") then ' is below blank?
          SetTile(x,y,asc("."))
          SetTile(x,y+1,tc)
          changed = 1
          lastfall = 1
        endif
      endif
    next x
  next y
  DoFalling = changed
end function

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Compare each tile to its neighbor right and down.
' If they match, mark them for fusion
function DoFusing()

  local integer tile

  'for y = 0 to PLAYHEIGHT : fusem(y)="............" : next y
  for y = 0 to PLAYHEIGHT : fusem(y)=SPACE$(PLAYWIDTH+1) : next y
  boom = 0
  for y = 1 to PLAYHEIGHT-1
    for x = 2 to PLAYWIDTH-1
      tile = GetTile(x,y)
      if tile > asc("0") and tile < asc("9") then
        if tile = GetTile(x+1,y) then  ' check tile to the right
          mid$(fusem(y),x,1)) = "*"
          mid$(fusem(y),x+1,1)) = "*"
          boom = 1
        endif
        if tile = GetTile(x,y+1) then  ' check tile below
          mid$(fusem(y),x,1)) = "*"
          mid$(fusem(y+1),x,1)) = "*"
          boom = 1
        endif
      endif
    next x
  next y

  if boom then    ' matching neighbours so make things go boom
    play stop   ' stop possible previous phwoosh or clink
    play wav "Pop.wav"
    for a = 1 to 3  ' animate explosion frames
      for y = 1 to PLAYHEIGHT-1
        for x = 2 to PLAYWIDTH-1
          ' show explosion animation image
          if mid$(fusem(y),x,1) = "*" then
            blit (8+a)*30,30,XOFFSET+(x+7)*30,YOFFSET+y*30,30,30,2
            SetTile(x,y,asc("."))  ' erase the tile
          endif
        next x
      next y
      ' display screen
      page copy 1 to 0,B
      pause 100
    next a
    DrawPlayingArea
    page copy 1 to 0,B
  endif
  DoFusing = boom
end function

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
sub DrawCurrentLevel

  DrawPlayingArea

  ' erase previous level numeral
  DrawTile2(4, 30,5*30)   ' metal tile background
  DrawTile2(4, 60,5*30)
  DrawTile2(4, 90,5*30)
  DrawTile2(4,120,5*30)
  ' draw new level numeral
  text 90+XOFFSET, 150+YOFFSET, str$(LEVEL), "CT" , 5, 1, &hFFB000, -1
  text 88+XOFFSET, 152+YOFFSET, str$(LEVEL), "CT" , 5, 1, &hFFB000, -1
  text 92+XOFFSET, 152+YOFFSET, str$(LEVEL), "CT" , 5, 1, &hFFB000, -1
  text 90+XOFFSET, 154+YOFFSET, str$(LEVEL), "CT" , 5, 1, &hFFB000, -1

  d = DoPhysics()   ' in case level isn't stable at start

end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Draw the current playing area as held in the 'currentlevel' array
' Also updates 'tilecount' with the number of tiles remaining
sub DrawPlayingArea
  tilecount = 0
  for y = 0 to PLAYHEIGHT
    for x = 1 to PLAYWIDTH
      t = GetTile(x,y)
      if t > asc("0") and t < asc("9") then tilecount = tilecount + 1
      DrawTile(t,(x+7)*30,y*30)
    next x
  next y
end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Return the asc value of the tile at the given coordinates
' Valid tiles are "." "B" "C" "1" to "8"
' Returns 0 for invalid request
function GetTile(x,y)
  if x > 0 and x <= PLAYWIDTH and y >= 0 and y <= PLAYHEIGHT then
    GetTile = asc(mid$(currentlevel(y),x,1))
  else
    GetTile = 0
  endif
end function

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Set the tile at the given coordinates
sub SetTile(x,y,tile)
  if x > 0 and x <= PLAYWIDTH and y >= 0 and y <= PLAYHEIGHT then
    mid$(currentlevel(y),x,1)) = chr$(tile)
  endif
end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' fill the screen with bricks - the background image
sub DrawBrickWall
  for y = 0 to PLAYHEIGHT
    for x = 0 to PLAYWIDTH + 7
      DrawTile(asc("B"),x*30,y*30)
    next x
  next y
end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' bolt on a sheet of metal for displaying the level
sub DrawMetalSign
  DrawTile2(0,  0,3*30)
  DrawTile2(1, 30,3*30)
  DrawTile2(1, 60,3*30)
  DrawTile2(1, 90,3*30)
  DrawTile2(1,120,3*30)
  DrawTile2(2,150,3*30)

  for i = 4 to 5
    DrawTile2(3,  0,i*30)
    DrawTile2(4, 30,i*30)
    DrawTile2(4, 60,i*30)
    DrawTile2(4, 90,i*30)
    DrawTile2(4,120,i*30)
    DrawTile2(5,150,i*30)
  next i

  DrawTile2(6,  0,6*30)
  DrawTile2(7, 30,6*30)
  DrawTile2(7, 60,6*30)
  DrawTile2(7, 90,6*30)
  DrawTile2(7,120,6*30)
  DrawTile2(8,150,6*30)

  ' add level text
  text 90+XOFFSET, 110+YOFFSET, "LEVEL", "CT" , 5, 1, &hFFB000, -1
  text 88+XOFFSET, 112+YOFFSET, "LEVEL", "CT" , 5, 1, &hFFB000, -1
  text 92+XOFFSET, 112+YOFFSET, "LEVEL", "CT" , 5, 1, &hFFB000, -1
  text 90+XOFFSET, 114+YOFFSET, "LEVEL", "CT" , 5, 1, &hFFB000, -1
end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Draw the requested tile on the buffer at the given coordinates
' Each tile is 30x30 pixels
' Tiles are stored beside each other on page 2
sub DrawTile(num,x,y)

  if num then
    select case num
      case asc(".")    ' blank
        box XOFFSET+x,YOFFSET+y,30,30,1,0,0
      case asc("B")    ' brick
        blit 240,0,XOFFSET+x,YOFFSET+y,30,30,2
      case asc("C")    ' concrete block
        blit 270,0,XOFFSET+x,YOFFSET+y,30,30,2
      case else        ' playing tile
        blit (num-49)*30,0,XOFFSET+x,YOFFSET+y,30,30,2   ' asc("1") = 49
    end select
  endif
end sub

' Draw background tiles from the second row
sub DrawTile2(num,x,y)
  blit num*30,30,XOFFSET+x,YOFFSET+y,30,30,2
end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Load the requested level from the data statements
' restore command only seems to accept literals so a long select case is required
' Return 0 if no more levels, level number otherwise
function LoadLevel(level)

  if level > MAXLEVEL then
    LoadLevel = 0
    exit function
  endif

  for i = 0 to 12
    currentlevel(i) = levelsource(level,i)
  next i

  LoadLevel = level

end function

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Slurp all of the levels into RAM
sub ReadAllLevels

  restore LevelData   ' shift data pointer to start of level data
  for i = 1 to MAXLEVEL
    for j = 0 to PLAYHEIGHT
      read levelsource(i,j)
    next j
  next i

end sub

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
' Read the title data and paint it on the screen
' Attempt to mimic spray painting
sub PaintTitle(x,y,scale,colr)
  restore ff2string  ' move read pointer to data
  do
    read x1,y1
    do
      read x2,y2
      if x2 = 0 then exit do  ' stroke done
      PaintLine(x+x1*scale,y+y1*scale,x+x2*scale,y+y2*scale,colr)
      x1 = x2
      y1 = y2
    loop
    if y2 = 0 then exit do  ' all done
    pause y2
  loop
end sub

' Paint a line from x0,y0 to x1,y1
' Use Bresenham's algorithm and draw small circles
' for each position along the line
' Desired speed is slow, as if spray painting.
sub PaintLine(x0,y0,x1,y1,c)

  local integer dx, dy, stepx, stepy, frac

  x0 = int(x0) : y0 = int(y0)  
  x1 = int(x1) : y1 = int(y1)  
  dy = y1 - y0 : dx = x1 - x0 
  
  stepy = 1
  if dy < 0 then dy = -dy : stepy = -1

  stepx = 1
  if dx < 0 then dx = -dx : stepx = -1
  
  dy = dy * 2
  dx = dx * 2

  ' draw starting point
  circle x0, y0, SPRAYSIZE, SPRAYSIZE, 1, c 
  
  if(dx > dy) then ' shallow line = iterate over x
  frac = dy - dx / 2
  do while x0 <> x1
      if frac >= 0 then y0 = y0 + stepy : frac = frac - dx
      x0 = x0 + stepx
      frac = frac + dy
      circle x0, y0, SPRAYSIZE, SPRAYSIZE, 1, c
    pause STROKESPEED
  loop
  else  ' steep line = iterate over y
  frac = dx - dy / 2
  do while y0 <> y1
      if frac >= 0 then x0 = x0 + stepx : frac = frac - dy
      y0 = y0 + stepy
      frac = frac + dx
      circle x0, y0, SPRAYSIZE, SPRAYSIZE, 1, c
    pause STROKESPEED
  loop
  endif

end sub

' Stroke encoded lettering.
' Form characters by drawing lines point to point.
' End of stroke indicated by (0,n)
'   where n is delay (ms) before starting next stroke
' End of image indicated by (0,0)

ff2String:
' F  3 strokes
data 2.8750,2.0900
data 1.5000,6.0900
data 0,100

data 2.4899,2.1478
data 6.1250,1.6025
data 0,100

data 1.9600,4.1200
data 3.7500,3.8400
data 0,50
' A  3 strokes
data 5.9565,2.4139
data 3.3315,6.1639
data 0,50

data 5.9565,2.4139
data 6.3315,6.1639
data 0,50

data 4.1190,5.0389
data 6.8315,4.2889
data 0,50
' L  1 stroke
data 8.6200,2.1250
data 7.2500,5.8750
data 9.2500,5.3750
data 0,25
' F  3 strokes
data 11.1250,2.3750
data 9.8789,6.0000
data 0,100

data 10.5299,2.4643
data 13.4150,2.0315
data 0,100

data 10.4375,4.3750
data 12.0000,4.1250
data 0,50
' U
'ff2string:
data 14.1282,2.1600
data 13.2043,4.9230
data 13.1778,5.2580
data 13.2369,5.5379
data 13.3659,5.7640
data 13.6190,5.9120
data 13.9912,6.0116
data 14.3846,5.9359
data 14.8206,5.7437
data 15.2034,5.4195
data 15.4480,5.0750
data 15.5490,4.8317
data 16.4400,2.1600
data 0,1
' S
data 18.9423,3.1000
data 19.0823,2.8600
data 19.1023,2.6800
data 19.0623,2.4800
data 18.9423,2.2800
data 18.8100,2.1750
data 18.5850,2.1100
data 18.2623,2.0800
data 18.0023,2.1000
data 17.7223,2.2000
data 17.5023,2.3200
data 17.3023,2.5600
data 17.1340,2.8660
data 17.1023,3.1800
data 17.1640,3.4480
data 17.3623,3.7400
data 17.6150,3.9300
data 17.9823,4.1000
data 18.3623,4.2400
data 18.6480,4.3340
data 18.8623,4.5000
data 19.0023,4.7800
data 19.0023,5.0800
data 18.9423,5.3000
data 18.7823,5.5200
data 18.5680,5.7920
data 18.2900,5.9760
data 17.9460,6.1120
data 17.6350,6.1950
data 17.2300,6.1750
data 16.8700,5.9250
data 16.5600,5.6800
data 16.4600,5.1400
data 0,1
' 2
data 20.7080,2.2195
data 21.0979,1.9829
data 21.5382,1.8327
data 21.9692,1.7620
data 22.2894,1.7791
data 22.6645,1.8428
data 22.9464,2.0165
data 23.1372,2.2803
data 23.2462,2.5547
data 23.2643,2.9192
data 23.1936,3.3540
data 23.0749,3.6825
data 22.8056,4.0941
data 22.5127,4.3621
data 22.0272,4.7285
data 18.5500,7.4000
data 23.3515,6.6744
data 0,0     ' end of character data

'@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
'
' HEY! DON'T LOOK HERE! THAT'S CHEATING!
'
LevelData:
'************************
LevelData:
data "BBBBBBBBBBBBBBBB"
data "BBCCCCBBBBBCCCBB"
data "BC1..1CBBCC6..CB"
data "BBCCCCBBC5CCC.CB"
data "BBBBBBBBC6....CB"
data "BBCCCCBBC5CCCCBB"
data "BC2..2CBBCBBBBBB"
data "BCC.CCBBBBBCCCBB"
data "BBBCBBBBBBC..3CB"
data "BBCCCCCBBBBC.CBB"
data "BC4.4.4CBBC3.3CB"
data "BBCCC.CBBBBC3CBB"
data "BBBBBCBBBBBBCBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBCCCCCCCBBBBB"
data "BBBCC.....CBBBBB"
data "BBBC.....4CBBBBB"
data "BBBC....57CBBBBB"
data "BBBC...162CBBBBB"
data "BBBC..5623CBBBBB"
data "BBBC.4173CCBBBBB"
data "BBBCCCCCCCBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBCCCCCCBBBBBB"
data "BBBBC..23CBBBBBB"
data "BBBBC..67CBBBBBB"
data "BBBBCC.51CBBBBBB"
data "BBBBBC.4CCBBBBBB"
data "BBBBBC.CCBBBBBBB"
data "BBBBCC4.CCBBBBBB"
data "BBBBC56..CBBBBBB"
data "BBBBC72..CBBBBBB"
data "BBBBC13..CBBBBBB"
data "BBBBCCCCCCBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBCCCCBCCCCCBBBB"
data "BBC5.CCC.6.CBBBB"
data "BBC15.C.146CBBBB"
data "BBCC14.14CCCBBBB"
data "BBBCC545CCBBBBBB"
data "BBBBCCCCCBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBCCCCCCCCCCBBB"
data "BBBC.6.4....CBBB"
data "BBBC.1.6....CBBB"
data "BBBC.5.5.2.7CBBB"
data "BBBC.4.1.7.2CBBB"
data "BBBCCCCCCCCCCBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBCCCCBBBB"
data "BBBBBBBBC.4CBBBB"
data "BBBBBBBCC.5CBBBB"
data "BBBBBBCC.1CCBBBB"
data "BBBBBBC..CCBBBBB"
data "BBBBBCC..CBBBBBB"
data "BBBBCC..CCBBBBBB"
data "BBBBC45CCBBBBBBB"
data "BBBBC1.CBBBBBBBB"
data "BBBBCCCCBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBCCCCCCCCBBBB"
data "BBBBC46....CBBBB"
data "BBBBCCC.5..CBBBB"
data "BBBBC...C..CBBBB"
data "BBBBC...C.4CBBBB"
data "BBBBC61..CCCBBBB"
data "BBBBC165..6CBBBB"
data "BBBBCCCCCCCCBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBCCCBBBBBBB"
data "BBBBBC...CBBBBBB"
data "BBBBC1...4CBBBBB"
data "BBBBCC..5CCBBBBB"
data "BBBBCCC.CCCBBBBB"
data "BBBBCC...CCBBBBB"
data "BBBBC.....CBBBBB"
data "BBBBCC1.1CCBBBBB"
data "BBBBC.5.5.CBBBBB"
data "BBBBBC4.4CBBBBBB"
data "BBBBBBC4CBBBBBBB"
data "BBBBBBBCBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBCCCCCCCCCCBBB"
data "BBBC4..5.5..CBBB"
data "BBBC6..6.1..CBBB"
data "BBBC4541.5..CBBB"
data "BBBCCCCCCCCCCBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

' level 10
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBCCCCBBB"
data "BBBBBBBCBC.8CBBB"
data "BBBBBBC8CC.5CBBB"
data "BBBBBBC5..8CCBBB"
data "BBBBCCCCC.CBBBBB"
data "BBBCC.CCCCBBBBBB"
data "BBBC4.4CBBBBBBBB"
data "BBBC545CBBBBBBBB"
data "BBBCC5CCBBBBBBBB"
data "BBBBCCCBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBCCCCCCCCCCBBB"
data "BBBC........CBBB"
data "BBBC..235...CBBB"
data "BBBC..6C4...CBBB"
data "BBBC..CCC...CBBB"
data "BBBC3.7..565CBBB"
data "BBBC2C6..1C6CBBB"
data "BBBCCCC..CCCCBBB"
data "BBBC...454..CBBB"
data "BBBC...7C1..CBBB"
data "BBBC...CCC..CBBB"
data "BBBCCCCCCCCCCBBB"
data "BBBBBBBBBBBBBBBB"

data "BCCCCCCCCCCCCCCB"
data "C..............C"
data "C...........51.C"
data "C..CCCCC6CC.CC.C"
data "C.C....C36C.C..C"
data "C.C....C43C.C..C"
data "C.C....C54C.C..C"
data "C.C....C25C.C..C"
data "C..CCC.C4CC.CC.C"
data "C.......2......C"
data "C......45.1..1.C"
data "BCCCCCCCCCCCCCCB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BCCCCCCCCCCCCCCB"
data "BC...........1CB"
data "BC..4..8.....8CB"
data "BC..56.4..5..1CB"
data "BCCCCCCC.CCCCCCB"
data "BC............CB"
data "BC............CB"
data "BC............CB"
data "BC1....C..C.61CB"
data "BCCCCCCCCCCCCCCB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBCCCCCCCBBBB"
data "BBBBC....3..CBBB"
data "BBBC....CCCCBBBB"
data "BBBC......21CBBB"
data "BBBC...CCCCCBBBB"
data "BBBBC.2.1CBBBBBB"
data "BBBBC231CBBBBBBB"
data "BBBBBCCCBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBCCCCBBBBBBB"
data "BBBBC....CCCBBBB"
data "BBBBC.......CBBB"
data "BBBBC.....6.CBBB"
data "BBBC......5.CBBB"
data "BBBC5.....62CBBB"
data "BBBC6..CCCCCCBBB"
data "BBBBC.......CBBB"
data "BBBBC.......CBBB"
data "BBBBC.2.....CBBB"
data "BBBBBCCCCCCCBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBCCCCCCCCBBBB"
data "BBBCCCCCCCCCCBBB"
data "BBCCC......CCCBB"
data "BCCC......2.CCCB"
data "BBC..C4..5C..CBB"
data "BBC.C.C..C.C.CBB"
data "BBC..C....C..CBB"
data "BBC.....1....CBB"
data "BBC.....2....CBB"
data "BBC..C..1.C..CBB"
data "BBBC..CCCC..CBBB"
data "BBBBC..45..CBBBB"
data "BBBBBCCCCCCBBBBB"

data "BBBBBBBBCCCBBBBB"
data "BBBBBBBC..CBBBBB"
data "BBBBBBC..1CBBBBB"
data "BBBBCC...CCBBBBB"
data "BBBC1.....5CBBBB"
data "BBBC4.....CCBBBB"
data "BBBC3......2CBBB"
data "BBBC2......CCBBB"
data "BBBBC4C..2..CBBB"
data "BBBBC3C..1..CBBB"
data "BBBBC2C..C52CBBB"
data "BBBBBCCCCCCCBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBCCCCCCCBBBBB"
data "BBBC.1...1.CBBBB"
data "BCCCCCC.CCCCCCBB"
data "C54213...21345CB"
data "BCCCCC...CCCCCBB"
data "BBBBBC...CBBBBBB"
data "BBBBBC...CBBBBBB"
data "BBBBBC...CBBBBBB"
data "BBBBBC...CBBBBBB"
data "BBBBBC...CBBBBBB"
data "BBBBBC454CBBBBBB"
data "BBBBBBCCCBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BCCCCCCCCCCCCCCB"
data "BC....4.......CB"
data "BC....2...4...CB"
data "BC....1...1...CB"
data "BC...CCCCCCC..CB"
data "BC...CBBBBBC..CB"
data "BC...CBBBBBC..CB"
data "BC...CBBBBBC..CB"
data "BC...CBBBBBC..CB"
data "BC1..CBBBBBC2.CB"
data "BCCCCCBBBBBCCCCB"
data "BBBBBBBBBBBBBBBB"

' level 20
data "BBBBBBBBBBBBBBBB"
data "BBBBCCCCCCCBBBBB"
data "BBBC6....5.CBBBB"
data "BBC52...23.CBBBB"
data "BBBCC...CCCBBBBB"
data "BBC.1...6..CBBBB"
data "BBC32...13.CBBBB"
data "BBBCC...CCCBBBBB"
data "BBBBC...CBBBBBBB"
data "BBBBC...CBBBBBBB"
data "BBBBC1C.CBBBBBBB"
data "BBBBBCBCBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "CCCCCCCCCCCCCCCC"
data "C....23CC23....C"
data "C....CCCCCC....C"
data "C..............C"
data "C..............C"
data "C..............C"
data "C.....2323.....C"
data "C...61CCCC16...C"
data "C..61CBBBBC16..C"
data "C..1CBBBBBBC1..C"
data "C.6CBBBBBBBBC6.C"
data "C.CBBBBBBBBBBC.C"
data "CCBBBBBBBBBBBBCC"

data "BBBBBBBBBBBBBBBB"
data "BBCCCCCCCCCCCCBB"
data "BBC4231..4.82CBB"
data "CCCCCCC..CCCCCCC"
data "C.....C..C.....C"
data "C...6..........C"
data "C...1.C..C....6C"
data "C...8.CCCC....3C"
data "CCCCCCCBBCCCCCCC"
data "BBBBBBBBBBBBBBBB"
data "BBBCBBBBBBBBCBBB"
data "BBBCCCCCCCCCCBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBCCC"
data "BBBBBBBBBBBCC61C"
data "BBBBBBBBBBC4.2CB"
data "BBBBBBBBCC.21CBB"
data "BBBBBBBC..31CBBB"
data "BBBBBBC.43CCBBBB"
data "BBBBBC..2CBBBBBB"
data "BBBCC453CBBBBBBB"
data "BBC..21CBBBBBBBB"
data "BC..3CCBBBBBBBBB"
data "C.45CBBBBBBBBBBB"
data "C61CBBBBBBBBBBBB"
data "CCCBBBBBBBBBBBBB"

'level 24
data "BBBBCCCCCCCCBBBB"
data "BBBC..3.6...CBBB"
data "BBBC.24136..CBBB"
data "BBBC.41513..CBBB"
data "BBBC.123513.CBBB"
data "BBBC.CCCCCC.CBBB"
data "BBBC.CBBBBC.CBBB"
data "BBBC.CBBBBC.CBBB"
data "BBBC.CBBBBC.CBBB"
data "BBBC.CBBBBC.CBBB"
data "BBC..CBBBBC..CBB"
data "BBBC.CBBBBC.CBBB"
data "BBBBCBBBBBBCBBBB"

data "BBBBBBBBBBBBBBBB"
data "CCCCCCCCCCCCCCCC"
data "C...61.3..3.5..C"
data "CC.CCCCCCCCCCCCC"
data "C.......4.....CB"
data "CCCCCCCCCCCCC.CB"
data "C.....4...5...CB"
data "CC.CCCCCCCCCCCCB"
data "C......1..3...CB"
data "CCCCCCCCCCCCC.CB"
data "C6............CB"
data "CC6CCCCCCCCCCCCB"
data "BCCCBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BCBBBBBBBBBBBBCB"
data "BCCCCCCCCCCCCCCB"
data "BC6........CCCCB"
data "BC5..........1CB"
data "BC4..........2CB"
data "BC3.........13CB"
data "BC5.........CCCB"
data "BCC54.4612.2CCCB"
data "BCCCCCCCCCCCCCCB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BCCCCCCCCCCCCCBB"
data "BC..61...35..CBB"
data "BCCCCCC.CCCCCCBB"
data "BBC.514..34.CBBB"
data "BBCCCCC.CCCCCBBB"
data "BBBC61...35CBBBB"
data "BBBCCCC.CCCCBBBB"
data "BBBBC2...2CBBBBB"
data "BBBBCCC.CCCBBBBB"
data "BBBBBC...CBBBBBB"
data "BBBBBCC.CCBBBBBB"
data "BBBBBBC.CBBBBBBB"
data "BBBBBBCCCBBBBBBB"

data "BCCCCCCCCCCBBBBB"
data "C..........CBBBB"
data "C.1.2.5...6.CBBB"
data "C.C.C.C.C.C..CBB"
data "C.2.2.5.3.2.1.CB"
data "C.C.C.C.C.C.C.CB"
data "C.4.4.4.3.6.2.CB"
data "C.C.C.C.C.C.C.CB"
data "C.1.1.5.1.3.6.CB"
data "C.C.C.C.C.C.C.CB"
data "C.............CB"
data "C.C.C.C.C.C.C.CB"
data "BCCCCCCCCCCCCCBB"

data "BBBBBBCCCBBBBBBB"
data "BBBBBC.2.CBBBBBB"
data "BBBBBC.C.CBBBBBB"
data "BBBBC.2.1.CBBBBB"
data "BBBBC.C.C.CBBBBB"
data "BBBC.3.2.3.CBBBB"
data "BBBC.C.C.C.CBBBB"
data "BBC.1.5.5.4.CBBB"
data "BBC.C.C.C.C.CBBB"
data "BC.....1...5.CBB"
data "BC.C4C.C3C.C.CBB"
data "BCCCCCCCCCCCCCBB"
data "BBBBBBBBBBBBBBBB"

' level 30
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBCCCCCCCCCCCBB"
data "BBBC..5.426..CBB"
data "BBBC..C.25C..CBB"
data "BBBC...C5C...CBB"
data "BBBC...4C63..CBB"
data "BBBC..3C3C2..CBB"
data "BBBC..C.2.C..CBB"
data "BBBC...61.1.CCBB"
data "BBBCCCCCCCCCCCBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBCCCCCCCCCCCBBB"
data "BC.4.1.....4.CBB"
data "BC1C.C.C.C.C.CBB"
data "BBC.2.5.5.5.CBBB"
data "BBC2C.C.C.C.CBBB"
data "BBBC.3.3...CBBBB"
data "BBBC.C.C.C.CBBBB"
data "BBC.6.6.6...CBBB"
data "BBC.C.C.C.C.CBBB"
data "BC.......3...CBB"
data "BC.C.C.C.C.C.CBB"
data "BCCCCCCCCCCCCCBB"
data "BBBBBBBBBBBBBBBB"

data "CCCBBBBBBBBBBCCC"
data "C5.CCBBBBBBCC61C"
data "BC...CBBBBC452CB"
data "BBC4..CCCC.21CBB"
data "BBBC......31CBBB"
data "BBBBCC....CCBBBB"
data "BBBBBC.5..CBBBBB"
data "BBBCC4.3...CCBBB"
data "BBC..21CC....CBB"
data "BC..3CCBBCC...CB"
data "C.45CBBBBBBC...C"
data "C61CBBBBBBBBC..C"
data "CCCBBBBBBBBBBCCC"

data "BBBBBBBBBBBBBBBB"
data "BBBBCCCCCCCCCBCB"
data "BBBC1525251..C2C"
data "BCCCCCCCCCC..C3C"
data "C.......1241.C2C"
data "C..C.CCCCCCCCC3C"
data "C.CBCBBBBBBBBC2C"
data "C.CBBBBBBBBBBC3C"
data "C.CBBBBBBBBBBC2C"
data "C.CCCCCCCCCCCC3C"
data "C.1234152631542C"
data "C.4123415263154C"
data "BCCCCCCCCCCCCCCB"

data "BBBBBCCCCCCBBBBB"
data "BBBBCCCC.3CCBBBB"
data "BBBCCCC..CCCCBBB"
data "BBCCCCC3CCCCCCBB"
data "BCCC..CCC..CCCCB"
data "CCC.3..3...3CCCC"
data "CC..5..5..5CCCCC"
data "CC6.6..6..6CCCCC"
data "CC1.1..1..1CCCCC"
data "BCC.4..4..4.CCCB"
data "BBCC2.424.2CCCBB"
data "BBBCC.2C2.CCCBBB"
data "BBBBCCCCCCCCBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBCCCCCCCCCCCCBB"
data "BC............CB"
data "BC2.2.........CB"
data "BC5.1.1..6.6.6CB"
data "BC1.CCC..CCC.5CB"
data "BBC...C..C...CBB"
data "BBC...C..C...CBB"
data "BBC...C..C...CBB"
data "BBC1..C..C..5CBB"
data "BBBCC5C1.C.CCBBB"
data "BBBBBCCCCCCBBBBB"
data "BBBBBBBBBBBBBBBB"

function GenerateLevelCode(levnum) as float
  local float cc
  cc = 17e-12
'  for pri = 0 to 11
  for pri = 1 to 10
    cv$ = mid$(levelsource(levnum,6+5*cos(pi*pri/log((pri+7)>>1))),pri+3,1)
    cc = 10*cc + val(chr$(asc(cv$)-16*(asc(cv$)>62)))
  next pri
  GenerateLevelCode = cc
end function

'ReadAllLevels
'for i = 2 to 60
'  for j = 1 to i-1
'    glc1 = GenerateLevelCode(i)
'    glc2 = GenerateLevelCode(j)
'    if glc1 = glc2 then 
'      ? i, j, format$(glc1,"%12.0f")
'      '? i+30, format$(glc2,"%12.0f")
'    endif
'  next j
'next i
'end

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BCCCCCCCCCCCCCCB"
data "BC.....32...52CB"
data "BC1234.CC2.216CB"
data "BC51234CC16321CB"
data "BC45123CC62616CB"
data "BC34512CC23165CB"
data "BC23451CC32631CB"
data "BCCCCCCCCCCCCCCB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBCCCCCCCCCCCBBB"
data "BBC.....4656CBBB"
data "BBC.....3145CBBB"
data "BBC.....1232CBBB"
data "BBC.....CCCCCBBB"
data "BBC.....CBBBBBBB"
data "BBC.....CBBBBBBB"
data "BBC.....CBBBBBBB"
data "BBC.....CBBBBBBB"
data "BBCCC...CCCCCBBB"
data "BBBBCCC65432CBBB"
data "BBBBBBCCCCCCCBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBCCCCCCCCBBBB"
data "BBBC..5...4.CBBB"
data "BBBC.42..42.CBBB"
data "BBBC.245268.CBBB"
data "BBBC.CCCCCC.CBBB"
data "BBBC...C6...CBBB"
data "BBBC...6C...CBBB"
data "BBBC...C6...CBBB"
data "BBBC...6C...CBBB"
data "BBBC..8C6...CBBB"
data "BBBBCCCCCCCCBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBCCCCCCBBBBB"
data "BBBBC......CBBBB"
data "BBBC........CBBB"
data "BBC56.......6CBB"
data "BC345.......54CB"
data "BBC23......63CBB"
data "BBBC12.....2CBBB"
data "BBBBC1....1CBBBB"
data "BBBBBC....CBBBBB"
data "BBBBBBC..CBBBBBB"
data "BBBBBBBCCBBBBBBB"
data "BBBBBBBBBBBBBBBB"

' level 40
data "BBBBCCCCCCCCBBBB"
data "BBBC.35.5...CBBB"
data "BBBC.24135..CBBB"
data "BBBC.41613..CBBB"
data "BBBC.123613.CBBB"
data "BBBC.CCCCCC.CBBB"
data "BBBC.CBBBBC.CBBB"
data "BBBC.CBBBBC.CBBB"
data "BBBC.CBBBBC.CBBB"
data "BBBC.CBBBBC.CBBB"
data "BBC..CBBBBC..CBB"
data "BBBC.CBBBBC.CBBB"
data "BBBBCBBBBBBCBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBCCCCBB"
data "BBBBBBBBCC.52.CB"
data "BBBCCCCC436151CB"
data "BBC..2.45CCCCCBB"
data "BBC.1C.CCCBBBBBB"
data "BBC.3C.CCCBBBBBB"
data "BBBCCC...6CBBBBB"
data "BBBBBCC.CCBBBBBB"
data "BBBBBBC..CBBBBBB"
data "BBBBBBC.CBBBBBBB"
data "BBBBBBBCBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBCCCCCCCCCBBB"
data "BBBC...3.3.3.CBB"
data "BBC....CCCCCCCCB"
data "BC.......6.142.C"
data "BC.....CCCCCCCCC"
data "BC.....4..5..5.C"
data "BC.....CCCCCCCCC"
data "BC......1.....6C"
data "BC...C.CCCCCCCCC"
data "BC2C.C...CBBBBBB"
data "BC1C1C...CBBBBBB"
data "BCCCCCCCCCBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBCCCCCCCCCCCBB"
data "BBC354.456...6CB"
data "BBBCCCCCCC12.CCB"
data "BBBBBBBBBBCC.CBB"
data "BBBBBBBCCCC..CBB"
data "BBBBBBC3.....CBB"
data "BBBBBBBCCCCC.CBB"
data "BBBBBBBBBBBC.CBB"
data "BBBBBBBBBBBC.CBB"
data "BBBBBBBBBBBC1CBB"
data "BBBBBBBBBBBC2CBB"
data "BBBBBBBBBBBBCBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BCCCCCCCCCCCCCCB"
data "C5...........12C"
data "CCC..........CCC"
data "C43......6...31C"
data "CCC......1...CCC"
data "C.4......3....3C"
data "CCC......2....CC"
data "C45...C..1.C..1C"
data "CCC...CCCCCC..CC"
data "C.....3...6....C"
data "CCC.C.C.C.C.C.CC"
data "BBBCBCBCBCBCBCBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BCCCCCCCCCCCCCBB"
data "BC...........CBB"
data "BC.....4.....CBB"
data "BC....363....CBB"
data "BC...52312...CBB"
data "BC..5361652..CBB"
data "BC.324361356.CBB"
data "BC12436165356CBB"
data "BCCCCCCCCCCCCCBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBCCCCCCCBBBB"
data "BBBBC3421.6.CBBB"
data "BBBBBCCCC.CCBBBB"
data "BBBBC.6...65CBBB"
data "BBBBBCC.CCCCBBBB"
data "BBBBBC......CBBB"
data "BBBBBCCCC.CCBBBB"
data "BBBBC5.....5CBBB"
data "BBBBBCC.CCCCBBBB"
data "BBBBBBC...3.CBBB"
data "BBBBBBBCC342CBBB"
data "BBBBBBBBC421CBBB"
data "BBBBBBBBBCCCBBBB"

data "BBBCCCCCCCCCBBBB"
data "BBBC1.1.1..CBBBB"
data "BBBCCCCCCC.CBBBB"
data "BBBC2.2.24.CBBBB"
data "BBBCCCCCCC.CBBBB"
data "BBBC3.343..CBBBB"
data "BBBCCCCCCC.CBBBB"
data "BBBC..4.4..CBBBB"
data "BBBCCCCCCC.CBBBB"
data "BBBC..5.5..CBBBB"
data "BBBCCCCCCC.CBBBB"
data "BBBC..6.6..CBBBB"
data "BBBCCCCCCCCCBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBCCCCCCCCCCCCBB"
data "BBC.....1.243CBB"
data "BBC..3..2.451CBB"
data "BBC1.43.C.536CBB"
data "BBCCCCC.C.CCCCBB"
data "BBBC....C.2.CBBB"
data "BBBC...4C.6.CBBB"
data "BBBCCCCCCCCCCBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBCCCCCCCCBBBBB"
data "BBC........CBBBB"
data "BBC214.46...CBBB"
data "BBBCCCCCCCC..CBB"
data "BBC.......1..CBB"
data "BBC.......C..CBB"
data "BBC2....3C...CBB"
data "BBC3...3C....CBB"
data "BBC5..5C..C..CBB"
data "BBC1.5C4.4C16CBB"
data "BBBCCCCCCCCCCBBB"
data "BBBBBBBBBBBBBBBB"

' level 50
data "BBBCCCCCCCCCCBBB"
data "BBC1.321.....CBB"
data "BBC63215.5...CBB"
data "BBBCCCCC.C.61CBB"
data "BBBBBBBC.CCCCBBB"
data "BBBBBBBC.CBBBBBB"
data "BBBBBBBC.CBBBBBB"
data "BBBBBBBC.CBBBBBB"
data "BBBBBBBC.CBBBBBB"
data "BBBBBBC...CBBBBB"
data "BBBBBBC...CBBBBB"
data "BBBBBBBCCCBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBCCCCCBB"
data "BBBBBBBBC..434CB"
data "BBBBBBBC...CCCCB"
data "BBBBBBC....5.1CB"
data "BBBBBC.....CCCCB"
data "BBBBC......532CB"
data "BBBC.......CCCCB"
data "BBC...CC.C.2.4CB"
data "BC....C..C.CCCCB"
data "C....2C..C....CB"
data "C1...1CC3C.2.4CB"
data "BCCCCCCCCCCCCCBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBCCCCCCCCCCCBB"
data "BBBC....1..5.CBB"
data "BBBC....C..4.CBB"
data "BBBC..52C321.CBB"
data "BBBC.CCCCCCC.CBB"
data "BBBC...6C....CBB"
data "BBBC..24C....CBB"
data "BBBC..63.31..CBB"
data "BBBCCCCCCCCCCCBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBCCCCCCCCCCBB"
data "BBBBC2.123456CBB"
data "BBBBC3.CCCCCCCBB"
data "BBBBC4.4CBBBBBBB"
data "BBBBC1.1CBBBBBBB"
data "BBBBC6.5CBBBBBBB"
data "BBBBC5.3CBBBBBBB"
data "BBBBCC.2CBBBBBBB"
data "BBBBCC.6CBBBBBBB"
data "BBBBCC.CCBBBBBBB"
data "BBBBCCCCCBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBCCCCCCCCCCCCBB"
data "BC6.5.45.3.42.CB"
data "BBCCCCCCCCCCC..C"
data "BC.53.62.242.3.C"
data "C..CCCCCCCCCCCCB"
data "C...65....3.4.3C"
data "BCCCCCCCC.C.CCCB"
data "BBBBBBBBC3C.CBBB"
data "BBBBBBBBBCC.CBBB"
data "BBBBBBBBBBC.CBBB"
data "BBBCCCCCCC..3CBB"
data "BBC....54..6CCBB"
data "BBBCCCCCCCCCBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBCCCCCCCCCCCCBB"
data "BC...2...63.5.CB"
data "C.6C.C3.35C2C.3C"
data "C.2CCC1451CCC51C"
data "C.52.325231.636C"
data "C.63.254143.414C"
data "C121.631434.341C"
data "CCCCCCCCCCCCCCCC"
data "BBCBBCBBBBCBBCBB"
data "BBCBBCBBBBCBBCBB"
data "BCBBBBCBBCBBBBCB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBCCBBBBBBBCCBBB"
data "BC4.CBBBBBC.5CBB"
data "BC32CBBBBBC14CBB"
data "BBC42CBBBC.6CBBB"
data "BBC36CBBBC.5CBBB"
data "BBBC2.CBC..CBBBB"
data "BBBC6.CBC.5CBBBB"
data "BBBBC2.C..CBBBBB"
data "BBBBC3...5CBBBBB"
data "BBBBBC...CBBBBBB"
data "BBBBBC215CBBBBBB"
data "BBBBBBCCCBBBBBBB"

data "BBBCCCCCCCCCBBBB"
data "BBC6.6......CBBB"
data "BC1232......2CBB"
data "BBC323.4....3CBB"
data "BBBCCC.2....1CBB"
data "BBC....4...CCCBB"
data "BBC5...C.....CBB"
data "BBBC4..C..C4.CBB"
data "BBBBC..C..CCCCBB"
data "BBBBC.6C.....CBB"
data "BBBBC.5C.....CBB"
data "BBBBCCCCCCCCCBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BCCCCCCCCCCBBBBB"
data "BC23456...CBBBBB"
data "BCCCCCC...CBBBBB"
data "CCCCCCC...CCCCCC"
data "C.23456..1.1.1.C"
data "C.1C1C1C.C.C.C.C"
data "CCCCCCCCCCCCCCCC"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"
data "BBBBBBBBBBBBBBBB"

data "BBBBBCCCCCCCCCCB"
data "BBBBC3.434.4.4.C"
data "BBBBC13212.2123C"
data "BBBBC65121.1212C"
data "BBBCCCCCCC.CCC5C"
data "BBBC.......CBBCB"
data "BBBC6CC.CC.CBBBB"
data "BBBBCBC.CC.CBBBB"
data "BBBBBBC.CC.CBBBB"
data "BBBBBBC.CC.CBBBB"
data "BBBBBBC..C..CBBB"
data "BBBBBBC..C...CBB"
data "BBBBBBBCCBCCCBBB"

' level 60
data "BBBCCCCCCCCCBBBB"
data "BBC4523.....CBBB"
data "BBC5432..1.1CBBB"
data "BBCCCCC..CCCCCBB"
data "BBC..46..26...CB"
data "BBC..25..52..1CB"
data "BBC..CC..CC..CBB"
data "BBC..CC..CC..CBB"
data "BBC..CC..CC..CBB"
data "BBC..CC..CC..CBB"
data "BBC..CC..CC..CBB"
data "BBBCCBBCCBBCCBBB"
data "BBBBBBBBBBBBBBBB"

Level_Done:
data "0"


